/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.common.filesystem;

import ch.systemsx.cisd.base.exceptions.CheckedExceptionTunnel;
import ch.systemsx.cisd.base.exceptions.IOExceptionUnchecked;
import ch.systemsx.cisd.base.exceptions.InterruptedExceptionUnchecked;
import ch.systemsx.cisd.base.unix.Unix;
import ch.systemsx.cisd.base.utilities.OSUtilities;
import ch.systemsx.cisd.common.concurrent.IActivityObserver;
import ch.systemsx.cisd.common.concurrent.InactivityMonitor;
import ch.systemsx.cisd.common.concurrent.RecordingActivityObserverSensor;
import ch.systemsx.cisd.common.exceptions.ConfigurationFailureException;
import ch.systemsx.cisd.common.exceptions.EnvironmentFailureException;
import ch.systemsx.cisd.common.exceptions.FileExistsException;
import ch.systemsx.cisd.common.exceptions.UnknownLastChangedException;
import ch.systemsx.cisd.common.exceptions.UserFailureException;
import ch.systemsx.cisd.common.filesystem.IFileOverwriteStrategy;
import ch.systemsx.cisd.common.filesystem.IPathCopier;
import ch.systemsx.cisd.common.logging.ISimpleLogger;
import ch.systemsx.cisd.common.logging.LogCategory;
import ch.systemsx.cisd.common.logging.LogFactory;
import ch.systemsx.cisd.common.logging.LogLevel;
import ch.systemsx.cisd.common.parser.Line;
import ch.systemsx.cisd.common.parser.filter.AlwaysAcceptLineFilter;
import ch.systemsx.cisd.common.parser.filter.ILineFilter;
import ch.systemsx.cisd.common.process.ProcessExecutionHelper;
import ch.systemsx.cisd.common.process.ProcessResult;
import ch.systemsx.cisd.common.string.StringUtilities;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.RandomAccessFile;
import java.io.Reader;
import java.io.Serializable;
import java.io.StringWriter;
import java.io.Writer;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DurationFormatUtils;
import org.apache.log4j.Logger;

public final class FileUtilities {
    private static final Logger operationLog = LogFactory.getLogger(LogCategory.OPERATION, FileUtilities.class);
    private static final Logger machineLog = LogFactory.getLogger(LogCategory.MACHINE, FileUtilities.class);
    public static final FileFilter ACCEPT_ALL_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return true;
        }
    };
    public static final FileFilter ACCEPT_ALL_BUT_HIDDEN_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return !pathname.isHidden();
        }
    };
    private static final NumberFormat SIZE_FORMAT = new DecimalFormat("0.00");

    private FileUtilities() {
    }

    public static String loadToString(File file) throws IOExceptionUnchecked {
        String string;
        assert (file != null);
        FileReader fileReader = null;
        try {
            fileReader = new FileReader(file);
            string = FileUtilities.readString(new BufferedReader(fileReader));
        }
        catch (IOException ex) {
            try {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(fileReader);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((Reader)fileReader);
        return string;
    }

    public static byte[] loadToByteArray(File file) throws IOExceptionUnchecked {
        byte[] byArray;
        FileInputStream fstream;
        byte[] buffer;
        block7: {
            assert (file != null);
            int size = (int)file.length();
            if ((long)size != file.length()) {
                throw new IOExceptionUnchecked("File " + file.getPath() + " too large.");
            }
            buffer = new byte[size];
            fstream = null;
            fstream = new FileInputStream(file);
            int sizeRead = FileUtilities.fillBuffer(fstream, buffer);
            if (sizeRead >= size) break block7;
            byte[] smallerBuffer = new byte[sizeRead];
            System.arraycopy(buffer, 0, smallerBuffer, 0, sizeRead);
            byte[] byArray2 = smallerBuffer;
            IOUtils.closeQuietly((InputStream)fstream);
            return byArray2;
        }
        try {
            byArray = buffer;
        }
        catch (IOException ex) {
            try {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(fstream);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((InputStream)fstream);
        return byArray;
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static <T> T loadToObject(File file, Class<T> clazz) throws IOExceptionUnchecked, CheckedExceptionTunnel {
        Object object;
        assert (file != null);
        int size = (int)file.length();
        if ((long)size != file.length()) {
            throw new IOExceptionUnchecked("File " + file.getPath() + " too large.");
        }
        FileInputStream fstream = null;
        ObjectInputStream oistream = null;
        try {
            fstream = new FileInputStream(file);
            oistream = new ObjectInputStream(fstream);
            object = oistream.readObject();
        }
        catch (IOException ex) {
            try {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
                catch (ClassNotFoundException ex2) {
                    throw CheckedExceptionTunnel.wrapIfNecessary(ex2);
                }
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(oistream);
                IOUtils.closeQuietly((InputStream)fstream);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((InputStream)oistream);
        IOUtils.closeQuietly((InputStream)fstream);
        return (T)object;
    }

    private static int fillBuffer(InputStream input, byte[] buffer) throws IOException {
        int ofs = 0;
        int len = buffer.length;
        int count = 0;
        int n = 0;
        while (len > 0 && -1 != (n = input.read(buffer, ofs, len))) {
            ofs += n;
            len -= n;
            count += n;
        }
        return count;
    }

    public static String loadExactToString(File file) throws IOExceptionUnchecked {
        String string;
        assert (file != null);
        FileReader fileReader = null;
        try {
            fileReader = new FileReader(file);
            string = FileUtilities.readExactString(new BufferedReader(fileReader));
        }
        catch (IOException ex) {
            try {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(fileReader);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((Reader)fileReader);
        return string;
    }

    public static void writeToFile(File file, String str) throws IOExceptionUnchecked {
        assert (file != null) : "Unspecified file.";
        assert (str != null) : "Unspecified string.";
        FileWriter fileWriter = null;
        try {
            try {
                fileWriter = new FileWriter(file);
                fileWriter.write(str);
            }
            catch (IOException ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(fileWriter);
            throw throwable;
        }
        IOUtils.closeQuietly((Writer)fileWriter);
    }

    public static void writeToFile(File file, byte[] data) throws IOExceptionUnchecked {
        assert (file != null) : "Unspecified file.";
        assert (data != null) : "Unspecified data.";
        FileOutputStream fileStream = null;
        try {
            try {
                fileStream = new FileOutputStream(file);
                fileStream.write(data);
            }
            catch (IOException ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(fileStream);
            throw throwable;
        }
        IOUtils.closeQuietly((OutputStream)fileStream);
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static long writeToFile(File file, long dataPosition, InputStream dataInputStream) throws IOExceptionUnchecked {
        long l;
        BufferedInputStream dataBufferedInputStream;
        block12: {
            RandomAccessFile randomAccessFile = null;
            dataBufferedInputStream = null;
            byte[] dataBuffer = new byte[1024];
            try {
                randomAccessFile = new RandomAccessFile(file, "rw");
                randomAccessFile.seek(dataPosition);
                dataBufferedInputStream = new BufferedInputStream(dataInputStream);
                long dataSize = 0L;
                int dataPartSize = 0;
                while (true) {
                    if ((dataPartSize = dataBufferedInputStream.read(dataBuffer)) == -1) {
                        l = dataSize;
                        if (randomAccessFile == null) break block12;
                        break;
                    }
                    randomAccessFile.write(dataBuffer, 0, dataPartSize);
                    dataSize += (long)dataPartSize;
                }
            }
            catch (FileNotFoundException e) {
                try {
                    throw CheckedExceptionTunnel.wrapIfNecessary(e);
                    catch (IOException e2) {
                        throw CheckedExceptionTunnel.wrapIfNecessary(e2);
                    }
                }
                catch (Throwable throwable) {
                    if (randomAccessFile != null) {
                        try {
                            randomAccessFile.close();
                        }
                        catch (IOException iOException) {}
                    }
                    IOUtils.closeQuietly(dataBufferedInputStream);
                    throw throwable;
                }
            }
            try {
                randomAccessFile.close();
            }
            catch (IOException iOException) {}
        }
        IOUtils.closeQuietly((InputStream)dataBufferedInputStream);
        return l;
    }

    public static void writeToFile(File file, Serializable object) throws IOExceptionUnchecked {
        assert (file != null) : "Unspecified file.";
        assert (object != null) : "Unspecified data.";
        FileOutputStream fileStream = null;
        ObjectOutputStream oostream = null;
        try {
            try {
                fileStream = new FileOutputStream(file);
                oostream = new ObjectOutputStream(fileStream);
                oostream.writeObject(object);
            }
            catch (IOException ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(oostream);
            IOUtils.closeQuietly((OutputStream)fileStream);
            throw throwable;
        }
        IOUtils.closeQuietly((OutputStream)oostream);
        IOUtils.closeQuietly((OutputStream)fileStream);
    }

    public static void appendToFile(File file, String str, boolean shouldAppendNewline) throws IOExceptionUnchecked {
        assert (file != null) : "Unspecified file.";
        assert (str != null) : "Unspecified string.";
        BufferedWriter bw = null;
        FileWriter fw = null;
        try {
            try {
                fw = new FileWriter(file, true);
                bw = new BufferedWriter(fw);
                bw.append(str);
                if (shouldAppendNewline) {
                    bw.newLine();
                }
                bw.flush();
            }
            catch (IOException ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(fw);
            throw throwable;
        }
        IOUtils.closeQuietly((Writer)fw);
    }

    public static final List<String> loadToStringList(File file) throws IOExceptionUnchecked {
        return FileUtilities.loadToStringList(file, null);
    }

    public static final List<String> loadToStringList(File file, ILineFilter lineFilterOrNull) throws IOExceptionUnchecked {
        List<String> list;
        assert (file != null) : "Unspecified file.";
        FileReader fileReader = null;
        try {
            fileReader = new FileReader(file);
            list = FileUtilities.readStringList(new BufferedReader(fileReader), lineFilterOrNull);
        }
        catch (IOException ex) {
            try {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(fileReader);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((Reader)fileReader);
        return list;
    }

    public static String loadToString(Class<?> clazz, String resource) throws IOExceptionUnchecked {
        assert (clazz != null) : "Given class can not be null.";
        assert (resource != null && resource.length() > 0) : "Given resource can not be null.";
        BufferedReader reader = null;
        try {
            reader = FileUtilities.tryGetBufferedReader(clazz, resource);
            String string = reader == null ? null : FileUtilities.readString(reader);
            return string;
        }
        catch (IOException ex) {
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
        finally {
            IOUtils.closeQuietly((Reader)reader);
        }
    }

    public static final List<String> loadToStringList(Class<?> clazz, String resource) throws IOExceptionUnchecked {
        return FileUtilities.loadToStringList(clazz, resource, null);
    }

    public static final List<String> loadToStringList(Class<?> clazz, String resource, ILineFilter lineFilterOrNull) throws IOExceptionUnchecked {
        assert (clazz != null) : "Given class can not be null.";
        assert (StringUtils.isNotEmpty(resource)) : "Given resource can not be empty.";
        BufferedReader reader = null;
        try {
            reader = FileUtilities.tryGetBufferedReader(clazz, resource);
            List<String> list = reader == null ? null : FileUtilities.readStringList(reader, lineFilterOrNull);
            return list;
        }
        finally {
            IOUtils.closeQuietly((Reader)reader);
        }
    }

    private static final BufferedReader tryGetBufferedReader(Class<?> clazz, String resource) {
        InputStream stream = clazz.getResourceAsStream(resource);
        if (stream == null) {
            return null;
        }
        return new BufferedReader(new InputStreamReader(stream));
    }

    private static String readString(BufferedReader reader) throws IOException {
        String line;
        assert (reader != null) : "Unspecified BufferedReader.";
        StringBuilder builder = new StringBuilder();
        while ((line = reader.readLine()) != null) {
            builder.append(line).append('\n');
        }
        return builder.toString();
    }

    private static String readExactString(BufferedReader reader) throws IOException {
        assert (reader != null) : "Unspecified BufferedReader.";
        StringBuilder builder = new StringBuilder();
        int numRead = 0;
        while ((numRead = reader.read()) != -1) {
            builder.append(String.copyValueOf(Character.toChars(numRead)));
        }
        reader.close();
        return builder.toString();
    }

    public static final List<String> loadToStringList(InputStream is) throws IOExceptionUnchecked {
        return FileUtilities.readStringList(new BufferedReader(new InputStreamReader(is)), null);
    }

    private static final List<String> readStringList(BufferedReader reader, ILineFilter lineFilterOrNull) throws IOExceptionUnchecked {
        assert (reader != null) : "Unspecified BufferedReader.";
        ILineFilter lineFilter = lineFilterOrNull == null ? AlwaysAcceptLineFilter.INSTANCE : lineFilterOrNull;
        ArrayList<String> list = new ArrayList<String>();
        try {
            String line = reader.readLine();
            int lineNumber = 0;
            while (line != null) {
                if (lineFilter.acceptLine(new Line(lineNumber, line))) {
                    list.add(line);
                }
                ++lineNumber;
                line = reader.readLine();
            }
            return list;
        }
        catch (IOException ex) {
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
    }

    public static String checkPathFullyAccessible(File path, String kindOfPath) {
        assert (path != null);
        assert (kindOfPath != null);
        return FileUtilities.checkPathAccessible(path, kindOfPath, "path", true);
    }

    public static String checkPathReadAccessible(File path, String kindOfPath) {
        assert (path != null);
        assert (kindOfPath != null);
        return FileUtilities.checkPathAccessible(path, kindOfPath, "path", false);
    }

    public static String checkDirectoryReadAccessible(File directory, String kindOfDirectory) {
        assert (directory != null);
        assert (kindOfDirectory != null);
        String msg = FileUtilities.checkPathAccessible(directory, kindOfDirectory, "directory", false);
        if (msg == null && !directory.isDirectory()) {
            return String.format("Path '%s' is supposed to be a %s directory but isn't.", directory.getPath(), kindOfDirectory);
        }
        return msg;
    }

    public static String checkDirectoryFullyAccessible(File directory, String kindOfDirectory) {
        assert (directory != null);
        assert (kindOfDirectory != null);
        String msg = FileUtilities.checkPathAccessible(directory, kindOfDirectory, "directory", true);
        if (msg == null && !directory.isDirectory()) {
            return String.format("Path '%s' is supposed to be a %s directory but isn't.", directory.getPath(), kindOfDirectory);
        }
        return msg;
    }

    public static String checkFileReadAccessible(File file, String kindOfFile) {
        assert (file != null);
        assert (kindOfFile != null);
        String msg = FileUtilities.checkPathAccessible(file, kindOfFile, "directory", false);
        if (msg == null && !file.isFile()) {
            return String.format("Path '%s' is supposed to be a %s file but isn't.", file.getPath(), kindOfFile);
        }
        return msg;
    }

    public static String checkFileFullyAccessible(File file, String kindOfFile) {
        assert (file != null);
        assert (kindOfFile != null);
        String msg = FileUtilities.checkPathAccessible(file, kindOfFile, "file", true);
        if (msg == null && !file.isFile()) {
            return String.format("Path '%s' is supposed to be a %s file but isn't.", file.getPath(), kindOfFile);
        }
        return msg;
    }

    private static String checkPathAccessible(File path, String kindOfPath, String directoryOrFile, boolean readAndWrite) {
        assert (path != null);
        assert (kindOfPath != null);
        assert (directoryOrFile != null);
        if (!path.canRead()) {
            if (!path.exists()) {
                return String.format("%s %s '%s' does not exist.", StringUtilities.capitalize(kindOfPath), directoryOrFile, path.getPath());
            }
            return String.format("%s %s '%s' is not readable.", StringUtilities.capitalize(kindOfPath), directoryOrFile, path.getPath());
        }
        if (readAndWrite && !path.canWrite()) {
            return String.format("%s directory '%s' is not writable.", StringUtilities.capitalize(kindOfPath), path.getPath());
        }
        return null;
    }

    public static boolean deleteRecursively(File path) throws InterruptedExceptionUnchecked {
        assert (path != null);
        return FileUtilities.deleteRecursively(path, null, null);
    }

    public static boolean deleteRecursively(File path, ISimpleLogger loggerOrNull) throws InterruptedExceptionUnchecked {
        return FileUtilities.deleteRecursively(path, loggerOrNull, null);
    }

    public static boolean deleteRecursively(File path, ISimpleLogger loggerOrNull, IActivityObserver observerOrNull) throws InterruptedExceptionUnchecked {
        assert (path != null);
        if (path.isDirectory()) {
            if (FileUtilities.isSymbolicLink(path)) {
                return FileUtilities.deleteSymbolicLink(path, loggerOrNull);
            }
            FileUtilities.ensureWritable(path);
            File[] fileArray = path.listFiles();
            int n = fileArray.length;
            int n2 = 0;
            while (n2 < n) {
                File file = fileArray[n2];
                if (Thread.interrupted()) {
                    throw new InterruptedExceptionUnchecked();
                }
                if (observerOrNull != null) {
                    observerOrNull.update();
                }
                if (file.isDirectory()) {
                    FileUtilities.deleteRecursively(file, loggerOrNull, observerOrNull);
                } else {
                    if (loggerOrNull != null) {
                        loggerOrNull.log(LogLevel.INFO, String.format("Deleting file '%s'", file.getPath()));
                    }
                    FileUtilities.delete(file);
                }
                ++n2;
            }
        }
        if (loggerOrNull != null) {
            loggerOrNull.log(LogLevel.INFO, String.format("Deleting directory '%s'", path.getPath()));
        }
        return FileUtilities.delete(path);
    }

    public static boolean isSymbolicLink(File path) {
        if (Unix.isOperational()) {
            return Unix.isSymbolicLink(path.getAbsolutePath());
        }
        return false;
    }

    private static boolean deleteSymbolicLink(File path, ISimpleLogger loggerOrNull) {
        if (loggerOrNull != null) {
            loggerOrNull.log(LogLevel.INFO, String.format("Deleting symbolic link to a directory '%s'", path.getPath()));
        }
        return FileUtilities.delete(path);
    }

    private static boolean ensureWritable(File path) {
        block3: {
            if (!path.canWrite() && Unix.isOperational()) {
                try {
                    Unix.setAccessMode(path.getPath(), (short)511);
                }
                catch (IOExceptionUnchecked ex) {
                    if (ex.getCause() == null || !ex.getCause().getMessage().contains("No such file or directory")) break block3;
                    return false;
                }
            }
        }
        return path.canWrite();
    }

    public static boolean delete(File file) {
        boolean OK = file.delete();
        if (OK) {
            return true;
        }
        if (file.exists() && Unix.isOperational()) {
            Unix.setAccessMode(file.getPath(), (short)511);
            return file.delete();
        }
        return false;
    }

    public static boolean deleteRecursively(File path, FileFilter filter, ISimpleLogger logger) throws InterruptedExceptionUnchecked {
        assert (path != null);
        return FileUtilities.deleteRecursively(path, filter, logger, null);
    }

    public static boolean deleteRecursively(File path, FileFilter filter, ISimpleLogger logger, IActivityObserver observerOrNull) throws InterruptedExceptionUnchecked {
        assert (path != null);
        assert (filter != null);
        if (filter.accept(path)) {
            return FileUtilities.deleteRecursively(path, logger, observerOrNull);
        }
        if (path.isDirectory()) {
            File[] fileArray = path.listFiles();
            int n = fileArray.length;
            int n2 = 0;
            while (n2 < n) {
                File file = fileArray[n2];
                if (Thread.interrupted()) {
                    throw new InterruptedExceptionUnchecked();
                }
                if (observerOrNull != null) {
                    observerOrNull.update();
                }
                FileUtilities.deleteRecursively(file, filter, logger, observerOrNull);
                ++n2;
            }
        }
        return false;
    }

    public static void findFiles(File path, List<File> files, FileFilter filter) throws InterruptedExceptionUnchecked {
        if (filter.accept(path)) {
            files.add(path);
        }
        if (path.isDirectory()) {
            File[] fileArray = path.listFiles();
            int n = fileArray.length;
            int n2 = 0;
            while (n2 < n) {
                File child = fileArray[n2];
                if (Thread.interrupted()) {
                    throw new InterruptedExceptionUnchecked();
                }
                FileUtilities.findFiles(child, files, filter);
                ++n2;
            }
        }
    }

    public static long lastChanged(File path, boolean subDirectoriesOnly, long stopWhenFindYounger) throws UnknownLastChangedException {
        return FileUtilities.lastChanged(path, subDirectoriesOnly, stopWhenFindYounger, null);
    }

    public static long lastChanged(File path, boolean subDirectoriesOnly, long stopWhenFindYounger, IActivityObserver observerOrNull) throws UnknownLastChangedException {
        return new LastChangedWorker(path, subDirectoriesOnly, stopWhenFindYounger, false, observerOrNull).getLastChanged();
    }

    public static long lastChangedRelative(File path, boolean subDirectoriesOnly, long stopWhenFindYoungerRelative) throws UnknownLastChangedException {
        return FileUtilities.lastChangedRelative(path, subDirectoriesOnly, stopWhenFindYoungerRelative, null);
    }

    public static long lastChangedRelative(File path, boolean subDirectoriesOnly, long stopWhenFindYoungerRelative, IActivityObserver observerOrNull) throws UnknownLastChangedException {
        return new LastChangedWorker(path, subDirectoriesOnly, stopWhenFindYoungerRelative, true, observerOrNull).getLastChanged();
    }

    public static long lastChanged(File path) throws UnknownLastChangedException {
        return FileUtilities.lastChanged(path, false, 0L);
    }

    public static final File removePrefixFromFileName(File file, String prefix) {
        assert (file != null);
        String name = file.getName();
        if (StringUtils.isEmpty(prefix)) {
            return file;
        }
        if (name.indexOf(prefix) < 0) {
            return file;
        }
        return new File(file.getParent(), name.substring(prefix.length()));
    }

    public static final File createNextNumberedFile(File path, Pattern regex) {
        return FileUtilities.createNextNumberedFile(path, regex, null);
    }

    public static final File createNextNumberedFile(File path, Pattern regexOrNull, String defaultFileNameOrNull) {
        assert (path != null);
        String filePath = path.getPath();
        String defaultPathNameOrNull = defaultFileNameOrNull == null ? null : new File(path.getParentFile(), defaultFileNameOrNull).getPath();
        String uniqueFilePath = StringUtilities.createUniqueString(filePath, new StringUtilities.IUniquenessChecker(){

            @Override
            public boolean isUnique(String str) {
                return !new File(str).exists();
            }
        }, regexOrNull, defaultPathNameOrNull);
        return new File(uniqueFilePath);
    }

    public static final String getRelativeFilePath(File root, File file) {
        String rootPath;
        assert (root != null) : "Given root can not be null.";
        assert (file != null) : "Given file can not be null.";
        String filePath = file.getAbsolutePath();
        if (filePath.equals(rootPath = root.getAbsolutePath())) {
            return "";
        }
        if (!rootPath.endsWith(File.separator)) {
            rootPath = String.valueOf(rootPath) + File.separator;
        }
        if (filePath.startsWith(rootPath)) {
            return filePath.substring(rootPath.length());
        }
        return null;
    }

    public static final String getRelativeFilePathOrDie(File root, File file) {
        String relativePath = FileUtilities.getRelativeFilePath(root, file);
        if (relativePath == null) {
            throw UserFailureException.fromTemplate("Directory %s should be a subdirectory of directory %s.", file, root);
        }
        return relativePath;
    }

    public static final String getParentRelativePath(String relativePath) {
        if (StringUtils.isBlank(relativePath)) {
            return null;
        }
        int lastIndexOf = relativePath.lastIndexOf(File.separator);
        if (lastIndexOf > -1) {
            return relativePath.substring(0, lastIndexOf);
        }
        return "";
    }

    public static final String getFileNameFromRelativePath(String relativePath) {
        int index = relativePath.lastIndexOf(47);
        return index < 0 ? relativePath : relativePath.substring(index + 1);
    }

    public static final boolean hasHDF5ContainerSuffix(File file) {
        return FilenameUtils.isExtension((String)file.getName().toLowerCase(), Arrays.asList("h5", "h5ar"));
    }

    public static final boolean isHDF5ContainerFile(File file) {
        return file.isFile() && FileUtilities.hasHDF5ContainerSuffix(file);
    }

    public static File[] tryListFiles(File directory, ISimpleLogger loggerOrNull) {
        return FileUtilities.tryListFiles(directory, ACCEPT_ALL_FILTER, loggerOrNull);
    }

    public static File[] tryListFiles(File directory, FileFilter filterOrNull, ISimpleLogger loggerOrNull) {
        File[] paths = null;
        RuntimeException ex = null;
        try {
            paths = directory.listFiles(filterOrNull);
        }
        catch (RuntimeException e) {
            ex = e;
        }
        if (paths == null && loggerOrNull != null) {
            FileUtilities.logFailureInDirectoryListing(ex, directory, loggerOrNull);
        }
        return paths;
    }

    public static final void sortByLastModified(File[] files) {
        Arrays.sort(files, LastModifiedFileComparator.LASTMODIFIED_COMPARATOR);
    }

    private static void logFailureInDirectoryListing(RuntimeException exOrNull, File directory, ISimpleLogger logger) {
        if (exOrNull == null) {
            if (directory.isFile()) {
                logger.log(LogLevel.ERROR, String.format("Failed to get listing of directory '%s' (path is file instead of directory).", directory));
            } else {
                logger.log(LogLevel.ERROR, String.format("Failed to get listing of directory '%s' (path not found).", directory));
            }
        } else {
            StringWriter exStackWriter = new StringWriter();
            exOrNull.printStackTrace(new PrintWriter(exStackWriter));
            logger.log(LogLevel.ERROR, String.format("Failed to get listing of directory '%s'. Exception: %s", directory, exStackWriter.toString()));
        }
    }

    public static final File[] listFiles(File directory) throws EnvironmentFailureException {
        File[] fileList = directory.listFiles();
        if (fileList == null) {
            throw EnvironmentFailureException.fromTemplate("Failed to get listing of directory '%s'", directory.getAbsolutePath());
        }
        return fileList;
    }

    public static List<File> listFiles(File directory, FileFilter filterOrNull, boolean recursive) throws EnvironmentFailureException {
        return FileUtilities.listFiles(directory, filterOrNull, recursive, null);
    }

    public static List<File> listFiles(File directory, FileFilter filterOrNull, boolean recursive, IActivityObserver observerOrNull) throws EnvironmentFailureException {
        return FileUtilities.listFiles(directory, filterOrNull, recursive, observerOrNull, null);
    }

    public static List<File> listFiles(File directory, FileFilter filterOrNull, boolean recursive, IActivityObserver observerOrNull, ISimpleLogger loggerOrNull) throws EnvironmentFailureException {
        assert (directory != null);
        LinkedList<File> result = new LinkedList<File>();
        FileUtilities.entryInternalListFiles(directory, result, filterOrNull == null ? new TrueFilter(observerOrNull) : filterOrNull, observerOrNull, recursive, FType.FILE, loggerOrNull);
        return result;
    }

    public static List<File> listFiles(File directory, String[] extensionsOrNull, boolean recursive) throws EnvironmentFailureException {
        return FileUtilities.listFiles(directory, extensionsOrNull, recursive, null, null);
    }

    public static List<File> listFiles(File directory, String[] extensionsOrNull, boolean recursive, IActivityObserver observerOrNull) throws EnvironmentFailureException {
        return FileUtilities.listFiles(directory, extensionsOrNull, recursive, observerOrNull, null);
    }

    public static List<File> listFiles(File directory, String[] extensionsOrNull, boolean recursive, IActivityObserver observerOrNull, ISimpleLogger loggerOrNull) throws EnvironmentFailureException {
        assert (directory != null);
        LinkedList<File> result = new LinkedList<File>();
        FileUtilities.entryInternalListFiles(directory, result, new ExtensionFileFilter(extensionsOrNull, recursive, observerOrNull), observerOrNull, recursive, FType.FILE, loggerOrNull);
        return result;
    }

    public static List<File> listDirectories(File directory, boolean recursive) throws EnvironmentFailureException {
        return FileUtilities.listDirectories(directory, recursive, null, null);
    }

    public static List<File> listDirectories(File directory, boolean recursive, IActivityObserver observerOrNull) throws EnvironmentFailureException {
        return FileUtilities.listDirectories(directory, recursive, observerOrNull, null);
    }

    public static List<File> listDirectories(File directory, boolean recursive, IActivityObserver observerOrNull, ISimpleLogger loggerOrNull) throws EnvironmentFailureException {
        assert (directory != null);
        LinkedList<File> result = new LinkedList<File>();
        FileUtilities.entryInternalListFiles(directory, result, new DirectoryFilter(observerOrNull), observerOrNull, recursive, FType.DIRECTORY, loggerOrNull);
        return result;
    }

    public static List<File> listFilesAndDirectories(File directory, boolean recursive) throws EnvironmentFailureException {
        return FileUtilities.listFilesAndDirectories(directory, recursive, null, null);
    }

    public static List<File> listFilesAndDirectories(File directory, boolean recursive, IActivityObserver observerOrNull) throws EnvironmentFailureException {
        return FileUtilities.listFilesAndDirectories(directory, recursive, observerOrNull, null);
    }

    public static List<File> listFilesAndDirectories(File directory, boolean recursive, IActivityObserver observerOrNull, ISimpleLogger loggerOrNull) throws EnvironmentFailureException {
        assert (directory != null);
        LinkedList<File> result = new LinkedList<File>();
        FileUtilities.entryInternalListFiles(directory, result, new TrueFilter(observerOrNull), observerOrNull, recursive, FType.EITHER, loggerOrNull);
        return result;
    }

    private static void entryInternalListFiles(File directory, List<File> result, FileFilter filter, IActivityObserver observerOrNull, boolean recursive, FType ftype, ISimpleLogger loggerOrNull) {
        if (recursive) {
            FileUtilities.internalListFiles(directory, result, new OrFilter(null, new DirectoryFilter(null), filter), observerOrNull, recursive, ftype, loggerOrNull);
        } else {
            FileUtilities.internalListFiles(directory, result, filter, observerOrNull, recursive, ftype, loggerOrNull);
        }
    }

    private static void internalListFiles(File directory, List<File> result, FileFilter filter, IActivityObserver observerOrNull, boolean recursive, FType ftype, ISimpleLogger loggerOrNull) {
        RuntimeException ex = null;
        File[] filteredFilesAndDirectories = null;
        try {
            filteredFilesAndDirectories = directory.listFiles(filter);
        }
        catch (RuntimeException e) {
            ex = e;
        }
        if (filteredFilesAndDirectories == null) {
            if (loggerOrNull != null) {
                FileUtilities.logFailureInDirectoryListing(ex, directory, loggerOrNull);
            }
            if (ex != null) {
                throw new EnvironmentFailureException("Error listing directory '" + directory + "'", ex);
            }
            throw new EnvironmentFailureException("Path '" + directory.getPath() + "' is not a directory.");
        }
        File[] fileArray = filteredFilesAndDirectories;
        int n = filteredFilesAndDirectories.length;
        int n2 = 0;
        while (n2 < n) {
            File f = fileArray[n2];
            if (observerOrNull != null) {
                observerOrNull.update();
            }
            if (f.isDirectory()) {
                if (ftype != FType.FILE && filter.accept(f)) {
                    result.add(f);
                }
                if (recursive) {
                    FileUtilities.internalListFiles(f, result, filter, observerOrNull, recursive, ftype, loggerOrNull);
                }
            } else if (ftype != FType.DIRECTORY && filter.accept(f)) {
                result.add(f);
            }
            ++n2;
        }
    }

    public static final File normalizeFile(File file) {
        assert (file != null) : "Given file can not be null.";
        try {
            return file.getCanonicalFile();
        }
        catch (IOException iOException) {
            return new File(FilenameUtils.normalize((String)file.getAbsolutePath()));
        }
    }

    public static final String getCanonicalPath(File file) {
        assert (file != null) : "Given file can not be null.";
        try {
            return file.getCanonicalPath();
        }
        catch (IOException iOException) {
            return file.getAbsolutePath();
        }
    }

    public static final String byteCountToDisplaySize(long size) {
        assert (size > -1L) : "Negative size value";
        String displaySize = size / 0x40000000L > 0L ? String.valueOf(SIZE_FORMAT.format((float)size / 1.0737418E9f)) + " GB" : (size / 0x100000L > 0L ? String.valueOf(SIZE_FORMAT.format((float)size / 1048576.0f)) + " MB" : (size / 1024L > 0L ? String.valueOf(SIZE_FORMAT.format((float)size / 1024.0f)) + " KB" : (size != 1L ? String.valueOf(size) + " bytes" : "1 byte")));
        return displaySize;
    }

    public static final String[] toFileNames(File[] files) {
        assert (files != null) : "Unspecified files";
        String[] fileNames = new String[files.length];
        int i = 0;
        File[] fileArray = files;
        int n = files.length;
        int n2 = 0;
        while (n2 < n) {
            File file = fileArray[n2];
            fileNames[i++] = file.getName();
            ++n2;
        }
        return fileNames;
    }

    public static void checkPathCopier(IPathCopier copier, String host, String rsyncExecutableOnHostOrNull, String rsyncModuleOrNull, String rsyncPasswordFileOrNull, long millisToWaitForCompletion) {
        if (rsyncModuleOrNull != null) {
            boolean connectionOK = copier.checkRsyncConnectionViaRsyncServer(host, rsyncModuleOrNull, rsyncPasswordFileOrNull, millisToWaitForCompletion);
            if (!connectionOK) {
                throw ConfigurationFailureException.fromTemplate("Connection to rsync module %s::%s failed", host, rsyncModuleOrNull);
            }
        } else {
            boolean connectionOK = copier.checkRsyncConnectionViaSsh(host, rsyncExecutableOnHostOrNull, millisToWaitForCompletion);
            if (!connectionOK) {
                throw ConfigurationFailureException.fromTemplate("No good rsync executable found on host '%s'", host);
            }
        }
    }

    public static void checkInputFile(File inFile) {
        if (!inFile.exists()) {
            throw CheckedExceptionTunnel.wrapIfNecessary(new FileNotFoundException("Input file '" + inFile.getAbsolutePath() + "' not found."));
        }
        if (!inFile.canRead()) {
            throw CheckedExceptionTunnel.wrapIfNecessary(new IOException("Cannot read input file '" + inFile.getAbsolutePath() + "'."));
        }
    }

    public static void checkOutputFile(File outFile, IFileOverwriteStrategy fileOverwriteStrategy) {
        if (outFile.exists() && !fileOverwriteStrategy.overwriteAllowed(outFile)) {
            throw CheckedExceptionTunnel.wrapIfNecessary(new FileExistsException(outFile));
        }
        if (outFile.exists()) {
            if (!outFile.canWrite()) {
                throw CheckedExceptionTunnel.wrapIfNecessary(new IOException("Cannot write to output file '" + outFile.getAbsolutePath() + "'."));
            }
        } else {
            File parent = outFile.getParentFile();
            if (parent != null && !FileUtilities.canCreateFile(parent)) {
                throw CheckedExceptionTunnel.wrapIfNecessary(new IOException("Cannot write to output directory '" + parent.getAbsolutePath() + "'."));
            }
        }
    }

    private static boolean canCreateFile(File directory) {
        if (!directory.isDirectory()) {
            return false;
        }
        if (directory.canWrite()) {
            return true;
        }
        if (OSUtilities.isWindows()) {
            try {
                File f = File.createTempFile("fcc", null, directory);
                f.delete();
                return true;
            }
            catch (IOException iOException) {
                return false;
            }
        }
        return false;
    }

    private static String getTempDir() {
        return System.getProperty("java.io.tmpdir");
    }

    public static boolean isValidFileName(String fileName) {
        if (StringUtils.isBlank(fileName) || fileName.contains(File.separator)) {
            return false;
        }
        File tmpFile = new File(FileUtilities.getTempDir(), fileName);
        try {
            if (tmpFile.createNewFile()) {
                tmpFile.delete();
                return true;
            }
        }
        catch (IOException ex) {
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
        return tmpFile.exists();
    }

    public static long getSizeOf(File file) {
        if (file == null) {
            throw new IllegalArgumentException("Unspecified file.");
        }
        if (!file.exists()) {
            throw new IllegalArgumentException("File does not exists: " + file);
        }
        if (file.isFile()) {
            return file.length();
        }
        if (!OSUtilities.isUnix()) {
            return FileUtils.sizeOfDirectory((File)file);
        }
        ArrayList<String> command = new ArrayList<String>(Arrays.asList("du", "-k", file.toString()));
        if (OSUtilities.isMacOS()) {
            command.add(2, "-d0");
        } else {
            command.add(2, "--max-depth");
            command.add(3, "0");
        }
        ProcessResult result = ProcessExecutionHelper.run(command, operationLog, machineLog);
        if (result.isOK()) {
            return Long.parseLong(result.getOutput().get(0).split("\\t")[0]) * 1024L;
        }
        Throwable exception = result.getProcessIOResult().tryGetException();
        if (exception != null) {
            throw CheckedExceptionTunnel.wrapIfNecessary(exception);
        }
        throw new EnvironmentFailureException("The size of the folder '" + file + "' couldn't be determined:\n command: " + result.getCommandName() + "\n commad line arguments: " + result.getCommandLine() + "\n output: " + result.getOutput() + "\n error output: " + result.getErrorOutput());
    }

    public static class DeleteActivityDetector
    extends RecordingActivityObserverSensor
    implements InactivityMonitor.IDescribingActivitySensor {
        private final File path;

        public DeleteActivityDetector(File path) {
            this.path = path;
        }

        @Override
        public String describeInactivity(long now) {
            return "No delete activity of path " + this.path.getPath() + " for " + DurationFormatUtils.formatDurationHMS(now - this.getLastActivityMillis());
        }
    }

    private static final class DirectoryFilter
    implements FileFilter {
        private final IActivityObserver observerOrNull;

        DirectoryFilter(IActivityObserver observerOrNull) {
            this.observerOrNull = observerOrNull;
        }

        @Override
        public boolean accept(File pathname) {
            if (this.observerOrNull != null) {
                this.observerOrNull.update();
            }
            return pathname.isDirectory();
        }
    }

    private static final class ExtensionFileFilter
    implements FileFilter {
        private final String[] extensionsOrNull;
        private final IActivityObserver observerOrNull;
        private final boolean recursive;

        ExtensionFileFilter(String[] extensionsOrNull, boolean recursive, IActivityObserver observerOrNull) {
            this.extensionsOrNull = extensionsOrNull;
            this.recursive = recursive;
            this.observerOrNull = observerOrNull;
        }

        private boolean correctType(File file) {
            return this.recursive || file.isFile();
        }

        private boolean match(String extensionFound) {
            if (this.extensionsOrNull == null) {
                return true;
            }
            if (extensionFound.length() == 0) {
                return false;
            }
            String[] stringArray = this.extensionsOrNull;
            int n = this.extensionsOrNull.length;
            int n2 = 0;
            while (n2 < n) {
                String ext = stringArray[n2];
                if (extensionFound.equalsIgnoreCase(ext)) {
                    return true;
                }
                ++n2;
            }
            return false;
        }

        @Override
        public boolean accept(File file) {
            if (this.observerOrNull != null) {
                this.observerOrNull.update();
            }
            if (this.recursive && file.isDirectory()) {
                return true;
            }
            return this.correctType(file) && this.match(FilenameUtils.getExtension((String)file.getName()));
        }
    }

    private static enum FType {
        FILE,
        DIRECTORY,
        EITHER;

    }

    private static final class LastChangedWorker {
        private final IActivityObserver observerOrNull;
        private final boolean subDirectoriesOnly;
        private final long reference;
        private final boolean referenceIsRelative;
        private long lastChanged;
        private boolean terminated;

        LastChangedWorker(File root, boolean subDirectoriesOnly, long reference, boolean referenceIsRelative, IActivityObserver observerOrNull) throws UnknownLastChangedException {
            assert (root != null);
            this.subDirectoriesOnly = subDirectoriesOnly;
            this.reference = reference;
            this.referenceIsRelative = referenceIsRelative;
            this.observerOrNull = observerOrNull;
            this.terminated = false;
            this.lastChanged = 0L;
            this.updateLastChanged(root);
            if (!this.terminated) {
                this.traverse(root);
            }
        }

        private void updateLastChanged(File path) throws UnknownLastChangedException {
            assert (path != null);
            long lastModified = path.lastModified();
            if (lastModified == 0L) {
                throw new UnknownLastChangedException(String.format("Can not get the last modification date of '%s'.", path.getPath()));
            }
            if (this.observerOrNull != null) {
                this.observerOrNull.update();
            }
            this.lastChanged = Math.max(lastModified, this.lastChanged);
            if (this.isYoungEnough(this.lastChanged)) {
                this.terminated = true;
            }
        }

        private boolean isYoungEnough(long currentLastChanged) {
            if (this.referenceIsRelative) {
                return this.reference > 0L && currentLastChanged > System.currentTimeMillis() - this.reference;
            }
            return this.reference > 0L && currentLastChanged > this.reference;
        }

        private void traverse(File path) throws UnknownLastChangedException {
            assert (path != null);
            if (!path.isDirectory()) {
                return;
            }
            File[] fileArray = this.getEntries(path);
            int n = fileArray.length;
            int n2 = 0;
            while (n2 < n) {
                File entry = fileArray[n2];
                this.updateLastChanged(entry);
                if (this.terminated) {
                    return;
                }
                InterruptedExceptionUnchecked.check();
                this.traverse(entry);
                ++n2;
            }
        }

        private File[] getEntries(File directory) {
            assert (directory != null);
            if (this.subDirectoriesOnly) {
                return directory.listFiles(new FileFilter(){

                    @Override
                    public boolean accept(File pathname) {
                        return pathname.isDirectory();
                    }
                });
            }
            return directory.listFiles();
        }

        long getLastChanged() {
            return this.lastChanged;
        }
    }

    private static final class OrFilter
    implements FileFilter {
        private final IActivityObserver observerOrNull;
        private final FileFilter filter1;
        private final FileFilter filter2;

        OrFilter(IActivityObserver observerOrNull, FileFilter filter1, FileFilter filter2) {
            this.observerOrNull = observerOrNull;
            this.filter1 = filter1;
            this.filter2 = filter2;
        }

        @Override
        public boolean accept(File pathname) {
            if (this.observerOrNull != null) {
                this.observerOrNull.update();
            }
            return this.filter1.accept(pathname) || this.filter2.accept(pathname);
        }
    }

    private static final class TrueFilter
    implements FileFilter {
        private final IActivityObserver observerOrNull;

        TrueFilter(IActivityObserver observerOrNull) {
            this.observerOrNull = observerOrNull;
        }

        @Override
        public boolean accept(File pathname) {
            if (this.observerOrNull != null) {
                this.observerOrNull.update();
            }
            return true;
        }
    }
}

