/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.hdf5.h5ar;

import ch.systemsx.cisd.base.exceptions.IErrorStrategy;
import ch.systemsx.cisd.base.exceptions.IOExceptionUnchecked;
import ch.systemsx.cisd.base.unix.FileLinkType;
import ch.systemsx.cisd.base.unix.Unix;
import ch.systemsx.cisd.hdf5.IHDF5Reader;
import ch.systemsx.cisd.hdf5.h5ar.ArchiveEntry;
import ch.systemsx.cisd.hdf5.h5ar.ArchiverException;
import ch.systemsx.cisd.hdf5.h5ar.ArchivingStrategy;
import ch.systemsx.cisd.hdf5.h5ar.GroupCache;
import ch.systemsx.cisd.hdf5.h5ar.IArchiveEntryProcessor;
import ch.systemsx.cisd.hdf5.h5ar.IArchiveEntryVisitor;
import ch.systemsx.cisd.hdf5.h5ar.IdCache;
import ch.systemsx.cisd.hdf5.h5ar.LinkRecord;
import ch.systemsx.cisd.hdf5.h5ar.UnarchivingException;
import ch.systemsx.cisd.hdf5.h5ar.Utils;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.zip.CRC32;
import ncsa.hdf.hdf5lib.exceptions.HDF5Exception;
import ncsa.hdf.hdf5lib.exceptions.HDF5JavaException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

class ArchiveEntryExtractProcessor
implements IArchiveEntryProcessor {
    private static final int ROOT_UID = 0;
    private final IArchiveEntryVisitor visitorOrNull;
    private final ArchivingStrategy strategy;
    private final File rootDirectory;
    private final String rootPathToStrip;
    private final byte[] buffer;
    private final GroupCache groupCache;

    ArchiveEntryExtractProcessor(IArchiveEntryVisitor visitorOrNull, ArchivingStrategy strategy, File rootDirectory, String rootPathToStrip, byte[] buffer) {
        this.visitorOrNull = visitorOrNull;
        this.strategy = strategy;
        this.rootDirectory = rootDirectory;
        String normalizedRootPathToStrip = Utils.normalizePath(rootPathToStrip);
        this.rootPathToStrip = "/".equals(normalizedRootPathToStrip) ? "" : normalizedRootPathToStrip;
        this.buffer = buffer;
        this.groupCache = new GroupCache();
    }

    @Override
    public boolean process(String dir, String path, LinkRecord link, IHDF5Reader reader, IdCache idCache, IErrorStrategy errorStrategy) throws IOException {
        if (this.strategy.doExclude(path, link.isDirectory())) {
            return false;
        }
        File file = this.createFile(path);
        if (link.isDirectory()) {
            if (file.exists() && !file.isDirectory()) {
                file.delete();
            }
            file.mkdirs();
            if (!file.isDirectory()) {
                errorStrategy.dealWithError(new UnarchivingException(file, new IOException("Failed to make directory '" + file.getAbsolutePath() + "'.")));
            }
            if (this.visitorOrNull != null) {
                this.visitorOrNull.visit(new ArchiveEntry(dir, path, link, idCache));
            }
        } else if (link.tryGetLinkTarget() != null && Unix.isOperational()) {
            try {
                file.delete();
                String linkTarget = link.tryGetLinkTarget();
                Unix.createSymbolicLink(linkTarget, file.getAbsolutePath());
                if (this.visitorOrNull != null) {
                    this.visitorOrNull.visit(new ArchiveEntry(dir, path, link, idCache));
                }
            }
            catch (IOExceptionUnchecked ex) {
                errorStrategy.dealWithError(new UnarchivingException(file, ex));
            }
        } else if (link.isSymLink()) {
            if (!Unix.isOperational()) {
                errorStrategy.warning("Warning: extracting symlink as regular file because Unix calls are not available on this system.");
            } else {
                errorStrategy.dealWithError(new UnarchivingException(path, (HDF5Exception)new HDF5JavaException("Symlink doesn't have a link target.")));
            }
        } else {
            try {
                long size = reader.object().getSize(path);
                int crc32 = this.copyFromHDF5(reader, path, size, file);
                this.restoreAttributes(file, link);
                FileSizeType sizeType = this.getFileSizeType(file);
                link.setVerifiedType(sizeType.type);
                link.setFileVerification(sizeType.size, crc32, file.lastModified() / 1000L);
                ArchiveEntry entry = new ArchiveEntry(dir, path, link, idCache);
                if (this.visitorOrNull != null) {
                    this.visitorOrNull.visit(entry);
                }
                if (!entry.isOK()) {
                    errorStrategy.dealWithError(new UnarchivingException(path, entry.getStatus(true)));
                }
            }
            catch (IOException ex) {
                errorStrategy.dealWithError(new UnarchivingException(file, ex));
            }
            catch (HDF5Exception ex) {
                errorStrategy.dealWithError(new UnarchivingException(path, ex));
            }
        }
        return true;
    }

    @Override
    public void postProcessDirectory(String dir, String path, LinkRecord link, IHDF5Reader reader, IdCache idCache, IErrorStrategy errorStrategy) throws IOException, HDF5Exception {
        File file = this.createFile(path);
        this.restoreAttributes(file, link);
    }

    private File createFile(String path) {
        String workingPath = path.startsWith(this.rootPathToStrip) ? path.substring(this.rootPathToStrip.length()) : path;
        File file = new File(this.rootDirectory, workingPath);
        return file;
    }

    private FileSizeType getFileSizeType(File file) {
        if (Unix.isOperational()) {
            Unix.Stat info = Unix.getLinkInfo(file.getPath(), false);
            return new FileSizeType(info.getLinkType(), info.getSize());
        }
        return new FileSizeType(file.isDirectory() ? FileLinkType.DIRECTORY : (file.isFile() ? FileLinkType.REGULAR_FILE : FileLinkType.OTHER), file.length());
    }

    private int copyFromHDF5(IHDF5Reader reader, String objectPath, long size, File destination) throws IOException {
        FileOutputStream output = FileUtils.openOutputStream(destination);
        CRC32 crc32 = new CRC32();
        try {
            long offset = 0L;
            while (offset < size) {
                int n = reader.opaque().readArrayToBlockWithOffset(objectPath, this.buffer, this.buffer.length, offset, 0);
                offset += (long)n;
                ((OutputStream)output).write(this.buffer, 0, n);
                crc32.update(this.buffer, 0, n);
            }
            ((OutputStream)output).close();
        }
        finally {
            IOUtils.closeQuietly(output);
        }
        return (int)crc32.getValue();
    }

    private void restoreAttributes(File file, LinkRecord linkInfoOrNull) {
        assert (file != null);
        if (linkInfoOrNull != null) {
            if (linkInfoOrNull.hasLastModified()) {
                file.setLastModified(linkInfoOrNull.getLastModified() * 1000L);
            }
            if (linkInfoOrNull.hasUnixPermissions() && Unix.isOperational()) {
                Unix.setAccessMode(file.getPath(), linkInfoOrNull.getPermissions());
                if (Unix.getUid() == 0) {
                    Unix.setOwner(file.getPath(), linkInfoOrNull.getUid(), linkInfoOrNull.getGid());
                } else if (this.groupCache.isUserInGroup(linkInfoOrNull.getGid())) {
                    Unix.setOwner(file.getPath(), Unix.getUid(), linkInfoOrNull.getGid());
                }
            }
        }
    }

    @Override
    public ArchiverException createException(String objectPath, String detailedMsg) {
        return new UnarchivingException(objectPath, detailedMsg);
    }

    @Override
    public ArchiverException createException(String objectPath, HDF5Exception cause) {
        return new UnarchivingException(objectPath, cause);
    }

    @Override
    public ArchiverException createException(String objectPath, RuntimeException cause) {
        return new UnarchivingException(objectPath, cause);
    }

    @Override
    public ArchiverException createException(File file, IOException cause) {
        return new UnarchivingException(file, cause);
    }

    private static class FileSizeType {
        final FileLinkType type;
        final long size;

        FileSizeType(FileLinkType type, long size) {
            this.type = type;
            this.size = size;
        }
    }
}

