/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.openbis.dss.generic.shared.content;

import ch.systemsx.cisd.base.exceptions.CheckedExceptionTunnel;
import ch.systemsx.cisd.common.exceptions.ConfigurationFailureException;
import ch.systemsx.cisd.common.exceptions.EnvironmentFailureException;
import ch.systemsx.cisd.common.filesystem.FileOperations;
import ch.systemsx.cisd.common.filesystem.FileUtilities;
import ch.systemsx.cisd.common.filesystem.IFileOperations;
import ch.systemsx.cisd.common.io.FullLengthReadingStream;
import ch.systemsx.cisd.common.logging.LogCategory;
import ch.systemsx.cisd.common.logging.LogFactory;
import ch.systemsx.cisd.common.properties.PropertyUtils;
import ch.systemsx.cisd.common.time.DateTimeUtils;
import ch.systemsx.cisd.common.utilities.ITimeProvider;
import ch.systemsx.cisd.common.utilities.SystemTimeProvider;
import ch.systemsx.cisd.openbis.dss.generic.shared.api.v1.IDssServiceRpcGeneric;
import ch.systemsx.cisd.openbis.dss.generic.shared.content.DelayedPersistenceManager;
import ch.systemsx.cisd.openbis.dss.generic.shared.content.DssServiceRpcGenericFactory;
import ch.systemsx.cisd.openbis.dss.generic.shared.content.IContentCache;
import ch.systemsx.cisd.openbis.dss.generic.shared.content.IDssServiceRpcGenericFactory;
import ch.systemsx.cisd.openbis.dss.generic.shared.content.IPersistenceManager;
import ch.systemsx.cisd.openbis.dss.generic.shared.content.SimpleFileBasePersistenceManager;
import ch.systemsx.cisd.openbis.dss.generic.shared.dto.DataSetPathInfo;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.IDatasetLocation;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import sun.net.www.protocol.https.Handler;

public class ContentCache
implements IContentCache,
InitializingBean {
    private static final int DEFAULT_MAX_WORKSPACE_SIZE = 1024;
    private static final String DEFAULT_CACHE_WORKSPACE_FOLDER = "../../data/dss-cache";
    private static final long DEFAULT_MINIMUM_KEEPING_TIME = 86400000L;
    public static final String CACHE_WORKSPACE_FOLDER_KEY = "cache-workspace-folder";
    public static final String CACHE_WORKSPACE_MAX_SIZE_KEY = "cache-workspace-max-size";
    public static final String CACHE_WORKSPACE_MIN_KEEPING_TIME_KEY = "cache-workspace-min-keeping-time";
    private static final Logger operationLog = LogFactory.getLogger(LogCategory.OPERATION, ContentCache.class);
    static final String CACHE_FOLDER = "cached";
    static final String DOWNLOADING_FOLDER = "downloading";
    static final String DATA_SET_INFOS_FILE = ".dataSetInfos";
    private static final Comparator<Map.Entry<String, DataSetInfo>> LAST_MODIFIED_COMPARATOR = new Comparator<Map.Entry<String, DataSetInfo>>(){

        @Override
        public int compare(Map.Entry<String, DataSetInfo> e1, Map.Entry<String, DataSetInfo> e2) {
            return (int)(e1.getValue().lastModified - e2.getValue().lastModified);
        }
    };
    private final LockManager fileLockManager;
    private final IDssServiceRpcGenericFactory serviceFactory;
    private final File workspace;
    private final ITimeProvider timeProvider;
    private final IFileOperations fileOperations;
    private final IPersistenceManager persistenceManager;
    private final long maxWorkspaceSize;
    private final long minimumKeepingTime;
    private HashMap<String, DataSetInfo> dataSetInfos;

    public static ContentCache create(Properties properties) {
        String workspacePath = properties.getProperty(CACHE_WORKSPACE_FOLDER_KEY, DEFAULT_CACHE_WORKSPACE_FOLDER);
        long maxWorkspaceSize = (long)PropertyUtils.getInt(properties, CACHE_WORKSPACE_MAX_SIZE_KEY, 1024) * 0x100000L;
        long minimumKeepingTimeInMillis = DateTimeUtils.getDurationInMillis(properties, CACHE_WORKSPACE_MIN_KEEPING_TIME_KEY, 86400000L);
        if (minimumKeepingTimeInMillis <= 60000L) {
            throw new EnvironmentFailureException("Minimum keeping time has to be a larger than a minute: " + minimumKeepingTimeInMillis);
        }
        File cacheWorkspace = new File(workspacePath);
        File dataSetInfosFile = new File(cacheWorkspace, DATA_SET_INFOS_FILE);
        DelayedPersistenceManager persistenceManager = new DelayedPersistenceManager(new SimpleFileBasePersistenceManager(dataSetInfosFile, "data set infos"));
        return new ContentCache(new DssServiceRpcGenericFactory(), cacheWorkspace, maxWorkspaceSize, minimumKeepingTimeInMillis, FileOperations.getInstance(), SystemTimeProvider.SYSTEM_TIME_PROVIDER, persistenceManager);
    }

    ContentCache(IDssServiceRpcGenericFactory serviceFactory, File cacheWorkspace, long maxWorkspaceSize, long minimumKeepingTime, IFileOperations fileOperations, ITimeProvider timeProvider, IPersistenceManager persistenceManager) {
        this.serviceFactory = serviceFactory;
        this.workspace = cacheWorkspace;
        this.maxWorkspaceSize = maxWorkspaceSize;
        this.minimumKeepingTime = minimumKeepingTime;
        this.fileOperations = fileOperations;
        this.timeProvider = timeProvider;
        this.persistenceManager = persistenceManager;
        this.fileLockManager = new LockManager();
        operationLog.info("Content cache created. Workspace: " + cacheWorkspace.getAbsolutePath());
    }

    public void afterPropertiesSet() {
        this.fileOperations.removeRecursivelyQueueing(new File(this.workspace, DOWNLOADING_FOLDER));
        int dataSetCount = this.initializeDataSetInfos();
        long totalSize = this.getTotalSize();
        operationLog.info("Content cache initialized. It contains " + FileUtilities.byteCountToDisplaySize(totalSize) + " from " + dataSetCount + " data sets.");
    }

    private int initializeDataSetInfos() {
        this.dataSetInfos = this.loadDataSetSize();
        File[] dataSetFolders = new File(this.workspace, CACHE_FOLDER).listFiles((FileFilter)DirectoryFileFilter.DIRECTORY);
        if (dataSetFolders == null) {
            return 0;
        }
        Arrays.sort(dataSetFolders, new Comparator<File>(){

            @Override
            public int compare(File f1, File f2) {
                return f1.getName().compareTo(f2.getName());
            }
        });
        boolean cachedFilesRemoved = false;
        File[] fileArray = dataSetFolders;
        int n = dataSetFolders.length;
        int n2 = 0;
        while (n2 < n) {
            File dataSetFolder = fileArray[n2];
            String dataSetCode = dataSetFolder.getName();
            DataSetInfo dataSetInfo = this.dataSetInfos.get(dataSetCode);
            if (dataSetInfo == null) {
                dataSetInfo = new DataSetInfo();
                dataSetInfo.lastModified = dataSetFolder.lastModified();
                dataSetInfo.size = FileUtilities.getSizeOf(dataSetFolder);
                operationLog.info("Data set info recreated for data set " + dataSetCode + ".");
                this.dataSetInfos.put(dataSetCode, dataSetInfo);
                cachedFilesRemoved = true;
            }
            ++n2;
        }
        if (cachedFilesRemoved) {
            this.persistenceManager.requestPersistence();
        }
        return dataSetFolders.length;
    }

    @Override
    public File getFile(String sessionToken, IDatasetLocation dataSetLocation, DataSetPathInfo path) {
        String pathInWorkspace = this.createPathInWorkspace(CACHE_FOLDER, dataSetLocation, path);
        this.fileLockManager.lock(pathInWorkspace);
        try {
            File file = new File(this.workspace, pathInWorkspace);
            if (!file.exists()) {
                this.downloadFile(sessionToken, dataSetLocation, path);
            } else {
                this.touchDataSetFolder(dataSetLocation.getDataSetCode());
            }
            this.persistenceManager.requestPersistence();
            File file2 = file;
            return file2;
        }
        finally {
            this.fileLockManager.unlock(pathInWorkspace);
        }
    }

    @Override
    public InputStream getInputStream(String sessionToken, IDatasetLocation dataSetLocation, DataSetPathInfo path) {
        String pathInWorkspace = this.createPathInWorkspace(CACHE_FOLDER, dataSetLocation, path);
        this.fileLockManager.lock(pathInWorkspace);
        File file = new File(this.workspace, pathInWorkspace);
        if (file.exists()) {
            try {
                FileInputStream fileInputStream = new FileInputStream(this.getFile(sessionToken, dataSetLocation, path));
                return fileInputStream;
            }
            catch (FileNotFoundException ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
            finally {
                this.fileLockManager.unlock(pathInWorkspace);
            }
        }
        try {
            File tempFile = this.createTempFile();
            InputStream inputStream = this.createInputStream(sessionToken, dataSetLocation, path);
            OutputStream fileOutputStream = this.createFileOutputStream(tempFile);
            return new ProxyInputStream(inputStream, fileOutputStream, dataSetLocation, tempFile, pathInWorkspace);
        }
        catch (Throwable t) {
            this.fileLockManager.unlock(pathInWorkspace);
            throw CheckedExceptionTunnel.wrapIfNecessary(t);
        }
    }

    private void downloadFile(String sessionToken, IDatasetLocation dataSetLocation, DataSetPathInfo path) {
        InputStream input = null;
        try {
            try {
                input = this.createInputStream(sessionToken, dataSetLocation, path);
                String pathInWorkspace = this.createPathInWorkspace(CACHE_FOLDER, dataSetLocation, path);
                File downloadedFile = this.createFileFromInputStream(input);
                this.moveDownloadedFileToCache(downloadedFile, pathInWorkspace, dataSetLocation.getDataSetCode());
            }
            catch (Exception ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(input);
            throw throwable;
        }
        IOUtils.closeQuietly((InputStream)input);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void moveDownloadedFileToCache(File downloadedFile, String pathInWorkspace, String dataSetCode) {
        File file = new File(this.workspace, pathInWorkspace);
        this.createFolder(file.getParentFile());
        boolean success = downloadedFile.renameTo(file);
        String msg = "'" + pathInWorkspace + "' successfully downloaded ";
        if (success) {
            this.touchDataSetFolder(dataSetCode);
            HashMap<String, DataSetInfo> hashMap = this.dataSetInfos;
            synchronized (hashMap) {
                DataSetInfo dataSetInfo = this.getDataSetInfo(dataSetCode);
                dataSetInfo.size += file.length();
                this.maintainCacheSize();
            }
            operationLog.debug(String.valueOf(msg) + "and successfully moved to cache.");
        } else {
            operationLog.warn(String.valueOf(msg) + "but couldn't move to cache.");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void touchDataSetFolder(String dataSetCode) {
        File dataSetFolder = new File(this.workspace, ContentCache.createDataSetPath(CACHE_FOLDER, dataSetCode));
        long lastModified = this.timeProvider.getTimeInMilliseconds();
        dataSetFolder.setLastModified(lastModified);
        HashMap<String, DataSetInfo> hashMap = this.dataSetInfos;
        synchronized (hashMap) {
            this.getDataSetInfo((String)dataSetCode).lastModified = lastModified;
        }
    }

    private InputStream createInputStream(String sessionToken, IDatasetLocation dataSetLocation, DataSetPathInfo path) {
        String dataStoreUrl = dataSetLocation.getDataStoreUrl();
        IDssServiceRpcGeneric service = this.serviceFactory.getService(dataStoreUrl);
        String dataSetCode = dataSetLocation.getDataSetCode();
        String relativePath = path.getRelativePath();
        URL url = this.createURL(service.getDownloadUrlForFileForDataSet(sessionToken, dataSetCode, relativePath));
        InputStream openStream = null;
        try {
            openStream = url.openStream();
            return new FullLengthReadingStream(openStream);
        }
        catch (IOException ex) {
            IOUtils.closeQuietly((InputStream)openStream);
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
    }

    private OutputStream createFileOutputStream(File file) {
        FileOutputStream outputStream = null;
        try {
            outputStream = new FileOutputStream(file);
            return outputStream;
        }
        catch (FileNotFoundException ex) {
            IOUtils.closeQuietly(outputStream);
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
    }

    private String createPathInWorkspace(String folder, IDatasetLocation dataSetLocation, DataSetPathInfo path) {
        String dataSetCode = dataSetLocation.getDataSetCode();
        return ContentCache.createDataSetPath(folder, String.valueOf(dataSetCode) + "/" + path.getRelativePath());
    }

    private static String createDataSetPath(String folder, String dataSetCode) {
        return String.valueOf(folder) + "/" + dataSetCode;
    }

    private URL createURL(String url) {
        try {
            if (url.toLowerCase().startsWith("https")) {
                return new URL(null, url, new Handler());
            }
            return new URL(url);
        }
        catch (MalformedURLException malformedURLException) {
            throw new ConfigurationFailureException("Malformed URL: " + url);
        }
    }

    private File createFileFromInputStream(InputStream inputStream) {
        File file;
        File file2 = this.createTempFile();
        FileOutputStream ostream = null;
        try {
            ostream = new FileOutputStream(file2);
            IOUtils.copyLarge((InputStream)inputStream, (OutputStream)ostream);
            file = file2;
        }
        catch (IOException ex) {
            try {
                file2.delete();
                throw CheckedExceptionTunnel.wrapIfNecessary(ex);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(ostream);
                throw throwable;
            }
        }
        IOUtils.closeQuietly((OutputStream)ostream);
        return file;
    }

    private File createTempFile() {
        File downLoadingFolder = new File(this.workspace, DOWNLOADING_FOLDER);
        this.createFolder(downLoadingFolder);
        try {
            File file = File.createTempFile("file-", null, downLoadingFolder);
            return file;
        }
        catch (IOException ex) {
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
    }

    private void createFolder(File folder) {
        boolean result;
        if (!folder.exists() && !(result = folder.mkdirs())) {
            throw new EnvironmentFailureException("Couldn't create folder: " + folder);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long getTotalSize() {
        HashMap<String, DataSetInfo> hashMap = this.dataSetInfos;
        synchronized (hashMap) {
            long sum = 0L;
            Collection<DataSetInfo> infos = this.dataSetInfos.values();
            for (DataSetInfo dataSetInfo : infos) {
                sum += dataSetInfo.size;
            }
            return sum;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DataSetInfo getDataSetInfo(String dataSetCode) {
        HashMap<String, DataSetInfo> hashMap = this.dataSetInfos;
        synchronized (hashMap) {
            DataSetInfo dataSetInfo = this.dataSetInfos.get(dataSetCode);
            if (dataSetInfo == null) {
                dataSetInfo = new DataSetInfo();
                this.dataSetInfos.put(dataSetCode, dataSetInfo);
            }
            return dataSetInfo;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maintainCacheSize() {
        ArrayList<Map.Entry<String, DataSetInfo>> entrySet;
        long totalSize = this.getTotalSize();
        if (totalSize < this.maxWorkspaceSize) {
            return;
        }
        HashMap<String, DataSetInfo> hashMap = this.dataSetInfos;
        synchronized (hashMap) {
            entrySet = new ArrayList<Map.Entry<String, DataSetInfo>>(this.dataSetInfos.entrySet());
        }
        Collections.sort(entrySet, LAST_MODIFIED_COMPARATOR);
        long nowMinusKeepingTime = this.timeProvider.getTimeInMilliseconds() - this.minimumKeepingTime;
        for (Map.Entry entry : entrySet) {
            DataSetInfo info = (DataSetInfo)entry.getValue();
            String dataSet = (String)entry.getKey();
            if (info.lastModified >= nowMinusKeepingTime || this.fileLockManager.isDataSetLocked(dataSet)) continue;
            File fileToRemove = new File(this.workspace, ContentCache.createDataSetPath(CACHE_FOLDER, dataSet));
            boolean success = this.fileOperations.removeRecursivelyQueueing(fileToRemove);
            if (success) {
                HashMap<String, DataSetInfo> hashMap2 = this.dataSetInfos;
                synchronized (hashMap2) {
                    this.dataSetInfos.remove(dataSet);
                }
                operationLog.info("Cached files for data set " + dataSet + " have been removed.");
                if ((totalSize -= info.size) >= this.maxWorkspaceSize) continue;
                break;
            }
            operationLog.error("Couldn't remove " + fileToRemove + ".");
        }
    }

    private HashMap<String, DataSetInfo> loadDataSetSize() {
        return (HashMap)this.persistenceManager.load(new HashMap());
    }

    static final class DataSetInfo
    implements Serializable {
        private static final long serialVersionUID = 1L;
        long lastModified;
        long size;

        DataSetInfo() {
        }

        public String toString() {
            return String.valueOf(this.size) + ":" + this.lastModified;
        }
    }

    private static final class LockManager {
        private final Map<String, LockWithCounter> locks = new HashMap<String, LockWithCounter>();

        private LockManager() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void lock(String path) {
            LockWithCounter lock;
            Map<String, LockWithCounter> map = this.locks;
            synchronized (map) {
                lock = this.locks.get(path);
                if (lock == null) {
                    lock = new LockWithCounter();
                    this.locks.put(path, lock);
                }
                LockWithCounter lockWithCounter = lock;
                lockWithCounter.count = lockWithCounter.count + 1;
            }
            lock.lock.lock();
        }

        synchronized void unlock(String path) {
            LockWithCounter lock = this.locks.get(path);
            if (lock != null) {
                lock.lock.unlock();
                LockWithCounter lockWithCounter = lock;
                int n = lockWithCounter.count - 1;
                lockWithCounter.count = n;
                if (n == 0) {
                    this.locks.remove(path);
                }
            }
        }

        synchronized boolean isDataSetLocked(String dataSetCode) {
            Set<String> keySet = this.locks.keySet();
            for (String key : keySet) {
                if (!key.startsWith(String.valueOf(ContentCache.createDataSetPath(ContentCache.CACHE_FOLDER, dataSetCode)) + "/")) continue;
                return true;
            }
            return false;
        }

        private static final class LockWithCounter {
            private Lock lock = new ReentrantLock();
            private int count;

            private LockWithCounter() {
            }
        }
    }

    private final class ProxyInputStream
    extends InputStream {
        private final InputStream inputStream;
        private final OutputStream fileOutputStream;
        private final IDatasetLocation dataSetLocation;
        private final File tempFile;
        private final String pathInWorkspace;
        private boolean closed;
        private boolean eof;

        private ProxyInputStream(InputStream inputStream, OutputStream fileOutputStream, IDatasetLocation dataSetLocation, File tempFile, String pathInWorkspace) {
            this.inputStream = inputStream;
            this.fileOutputStream = fileOutputStream;
            this.dataSetLocation = dataSetLocation;
            this.tempFile = tempFile;
            this.pathInWorkspace = pathInWorkspace;
        }

        @Override
        public int read() throws IOException {
            if (this.eof) {
                return -1;
            }
            int b = this.inputStream.read();
            if (b < 0) {
                this.eof = true;
            } else {
                this.fileOutputStream.write(b);
            }
            this.closeIfEndOfFile();
            return b;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            if (this.eof) {
                return -1;
            }
            int count = this.inputStream.read(b, off, len);
            if (count >= 0) {
                this.fileOutputStream.write(b, off, count);
            } else {
                this.eof = true;
            }
            this.closeIfEndOfFile();
            return count;
        }

        private void closeIfEndOfFile() throws IOException {
            if (this.eof) {
                this.close();
            }
        }

        @Override
        public void close() throws IOException {
            if (this.closed) {
                return;
            }
            this.inputStream.close();
            this.fileOutputStream.close();
            if (this.eof) {
                ContentCache.this.moveDownloadedFileToCache(this.tempFile, this.pathInWorkspace, this.dataSetLocation.getDataSetCode());
                ContentCache.this.persistenceManager.requestPersistence();
            } else {
                this.tempFile.delete();
            }
            this.closed = true;
            ContentCache.this.fileLockManager.unlock(this.pathInWorkspace);
        }

        protected void finalize() throws Throwable {
            if (!this.closed) {
                this.tempFile.delete();
                ContentCache.this.fileLockManager.unlock(this.pathInWorkspace);
            }
            super.finalize();
        }
    }
}

