/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.openbis.generic.shared.util;

import ch.rinn.restrictions.Private;
import ch.systemsx.cisd.base.exceptions.CheckedExceptionTunnel;
import ch.systemsx.cisd.common.exceptions.ConfigurationFailureException;
import ch.systemsx.cisd.common.filesystem.FileUtilities;
import ch.systemsx.cisd.common.filesystem.HostAwareFile;
import ch.systemsx.cisd.common.filesystem.IFreeSpaceProvider;
import ch.systemsx.cisd.common.logging.LogCategory;
import ch.systemsx.cisd.common.logging.LogFactory;
import ch.systemsx.cisd.common.utilities.ITimeProvider;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.WebClientConfiguration;
import ch.systemsx.cisd.openbis.generic.shared.util.ICacheManager;
import ch.systemsx.cisd.openbis.generic.shared.util.Key;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Serializable;
import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.SerializationUtils;
import org.apache.log4j.Logger;

public class CacheManager
implements ICacheManager {
    private static final int DAY = 86400000;
    @Private
    public static final String CACHE_FOLDER_KEY = "cache-folder";
    private static final String CACHE_FOLDER_DEFAULT_VALUE = "cache";
    @Private
    static final String MINIMUM_FREE_DISK_SPACE_KEY = "minimum-free-disk-space-in-MB";
    private static final int MINIMUM_FREE_DISK_SPACE_DEFAULT_VALUE = 1024;
    @Private
    static final String MAXIMUM_RETENTION_TIME_KEY = "maximum-retention-time-in-days";
    private static final int MAXIMUM_RETENTION_TIME_DEFAULT_VALUE = 7;
    @Private
    static final String CACHE_VERSION_FILE_NAME = ".cache-version";
    @Private
    static final String KEY_FILE_TYPE = ".key";
    @Private
    static final String DATA_FILE_TYPE = ".data";
    private static final Logger operationLog = LogFactory.getLogger(LogCategory.OPERATION, CacheManager.class);
    private final ITimeProvider timeProvider;
    private final Map<Key, FileName> keyToFileNameMap;
    private final IFreeSpaceProvider freeSpaceProvider;
    private final File cacheFolder;
    private final long minimumFreeDiskSpaceInKB;
    private final long maximumRetentionTimeInMillis;

    public CacheManager(WebClientConfiguration configuration, String technologyName, ITimeProvider timeProvider, IFreeSpaceProvider freeSpaceProvider, String cacheVersion) {
        this.timeProvider = timeProvider;
        this.freeSpaceProvider = freeSpaceProvider;
        this.keyToFileNameMap = new HashMap<Key, FileName>();
        this.cacheFolder = new File(this.getProperty(configuration, technologyName, CACHE_FOLDER_KEY, CACHE_FOLDER_DEFAULT_VALUE));
        this.minimumFreeDiskSpaceInKB = this.getIntegerProperty(configuration, technologyName, MINIMUM_FREE_DISK_SPACE_KEY, 1024) * 1024;
        this.maximumRetentionTimeInMillis = 86400000 * this.getIntegerProperty(configuration, technologyName, MAXIMUM_RETENTION_TIME_KEY, 7);
        String actualCacheVersion = this.tryToGetActualCacheVersion();
        if (cacheVersion.equals(actualCacheVersion)) {
            File[] keyFiles;
            File[] fileArray = keyFiles = this.cacheFolder.listFiles(new FilenameFilter(){

                @Override
                public boolean accept(File dir, String name) {
                    return name.endsWith(CacheManager.KEY_FILE_TYPE);
                }
            });
            int n = keyFiles.length;
            int n2 = 0;
            while (n2 < n) {
                File keyFile = fileArray[n2];
                String name = keyFile.getName();
                String fileName = name.substring(0, name.length() - KEY_FILE_TYPE.length());
                Key key = this.getKey(keyFile);
                this.keyToFileNameMap.put(key, new FileName(fileName));
                ++n2;
            }
        } else {
            FileUtilities.deleteRecursively(this.cacheFolder);
            this.cacheFolder.mkdirs();
            FileUtilities.writeToFile(new File(this.cacheFolder, CACHE_VERSION_FILE_NAME), cacheVersion);
        }
    }

    private String tryToGetActualCacheVersion() {
        File file = new File(this.cacheFolder, CACHE_VERSION_FILE_NAME);
        if (!file.exists()) {
            return null;
        }
        return FileUtilities.loadToString(file).trim();
    }

    private Key getKey(File keyFile) {
        Key key;
        try {
            key = (Key)SerializationUtils.deserialize((byte[])FileUtils.readFileToByteArray((File)keyFile));
        }
        catch (IOException ex) {
            throw CheckedExceptionTunnel.wrapIfNecessary(ex);
        }
        return key;
    }

    private int getIntegerProperty(WebClientConfiguration configuration, String technologyName, String key, int defaultValue) {
        String value = configuration.getPropertyOrNull(technologyName, key);
        if (value == null) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(value);
        }
        catch (NumberFormatException numberFormatException) {
            throw new ConfigurationFailureException("Web client configuration parameter '" + key + "' isn't a number: " + value);
        }
    }

    private String getProperty(WebClientConfiguration configuration, String technologyName, String key, String defaultValue) {
        String value = configuration.getPropertyOrNull(technologyName, key);
        return value == null ? defaultValue : value;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Object tryToGetData(Key key) {
        Map<Key, FileName> map = this.keyToFileNameMap;
        synchronized (map) {
            FileName fileName;
            block7: {
                fileName = this.keyToFileNameMap.get(key);
                if (fileName != null) break block7;
                return null;
            }
            File dataFile = new File(this.cacheFolder, fileName + DATA_FILE_TYPE);
            try {
                long t0 = this.timeProvider.getTimeInMilliseconds();
                byte[] bytes = FileUtils.readFileToByteArray((File)dataFile);
                Object object = SerializationUtils.deserialize((byte[])bytes);
                long duration = this.timeProvider.getTimeInMilliseconds() - t0;
                if (operationLog.isInfoEnabled()) {
                    operationLog.info(String.valueOf(duration) + " msec for retrieving cached data (" + bytes.length + " bytes) for key " + key + " from file " + dataFile);
                }
                fileName.touch(this.cacheFolder, this.timeProvider);
                return object;
            }
            catch (IOException ex) {
                operationLog.warn("Couldn't read data file '" + fileName + "' key " + key + ": " + ex.toString());
                return null;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void storeData(Key key, Object object) {
        Map<Key, FileName> map = this.keyToFileNameMap;
        synchronized (map) {
            FileName fileName = new FileName(this.timeProvider);
            try {
                if (!(object instanceof Serializable)) {
                    operationLog.warn("Can not store unserializable data for key " + key + ": " + object);
                } else {
                    long t0 = this.timeProvider.getTimeInMilliseconds();
                    this.cleanUp();
                    this.save(fileName + KEY_FILE_TYPE, key);
                    File dataFile = this.save(fileName + DATA_FILE_TYPE, (Serializable)object);
                    this.keyToFileNameMap.put(key, fileName);
                    long duration = this.timeProvider.getTimeInMilliseconds() - t0;
                    if (operationLog.isInfoEnabled()) {
                        operationLog.info(String.valueOf(duration) + " msec for storing data for key " + key + " in file " + dataFile);
                    }
                }
            }
            catch (Exception exception) {}
        }
    }

    private void cleanUp() {
        long currentTime = this.timeProvider.getTimeInMilliseconds();
        for (Map.Entry<Key, FileName> entry : this.keyToFileNameMap.entrySet()) {
            long time = entry.getValue().getTimeStamp().getTime();
            if (time + this.maximumRetentionTimeInMillis >= currentTime) continue;
            this.removeFromCache(entry.getKey());
        }
        HostAwareFile file = new HostAwareFile(this.cacheFolder);
        try {
            while (!this.keyToFileNameMap.isEmpty() && this.freeSpaceProvider.freeSpaceKb(file) < this.minimumFreeDiskSpaceInKB) {
                long oldestTimeStamp = Long.MAX_VALUE;
                Key oldestKey = null;
                for (Map.Entry<Key, FileName> entry : this.keyToFileNameMap.entrySet()) {
                    long time = entry.getValue().getTimeStamp().getTime();
                    if (time >= oldestTimeStamp) continue;
                    oldestTimeStamp = time;
                    oldestKey = entry.getKey();
                }
                this.removeFromCache(oldestKey);
            }
        }
        catch (IOException ex) {
            operationLog.warn("Can not obtain available free disk space.", ex);
        }
    }

    private void removeFromCache(Key key) {
        FileName fileName = this.keyToFileNameMap.remove(key);
        if (fileName == null) {
            return;
        }
        this.deleteFile(new File(this.cacheFolder, fileName + KEY_FILE_TYPE), key);
        this.deleteFile(new File(this.cacheFolder, fileName + DATA_FILE_TYPE), key);
    }

    private void deleteFile(File file, Key key) {
        boolean result = file.delete();
        if (result) {
            if (operationLog.isInfoEnabled()) {
                operationLog.info("For key " + key + " successfully removed from cache: " + file);
            }
        } else {
            operationLog.warn("For key " + key + " removing from cache caused some unknown error: " + file);
        }
    }

    private void assertFreeSpaceAvailableFor(int numberOfBytes) throws IOException {
        long freeSpace = 1024L * this.freeSpaceProvider.freeSpaceKb(new HostAwareFile(this.cacheFolder));
        if ((long)numberOfBytes > freeSpace) {
            throw new IOException("Can not store " + numberOfBytes + " because only " + freeSpace + " bytes are available on disk.");
        }
    }

    private File save(String fileName, Serializable object) {
        File file = new File(this.cacheFolder, fileName);
        try {
            byte[] bytes = SerializationUtils.serialize((Serializable)object);
            this.assertFreeSpaceAvailableFor(bytes.length);
            FileUtils.writeByteArrayToFile((File)file, (byte[])bytes);
            return file;
        }
        catch (Throwable t) {
            file.delete();
            operationLog.warn("Caching error: Couldn't save '" + file + "': " + t, t);
            throw CheckedExceptionTunnel.wrapIfNecessary(t);
        }
    }

    private static final class FileName {
        private static final String DELIMITER = "-";
        private static final String TIME_STAMP_FORMAT = "yyyyMMddHHmmssSSS";
        private static final String FILE_NAME_FORMAT = "{0,date,yyyyMMddHHmmssSSS}-{1}";
        private Date timeStamp;
        private String fileName;
        private int counter;

        FileName(ITimeProvider timeProvider) {
            this.createTimeStampAndFileName(timeProvider);
        }

        FileName(String fileName) {
            this.fileName = fileName;
            String[] fileNameParts = fileName.split(DELIMITER);
            if (fileNameParts.length != 2) {
                throw new IllegalArgumentException("Missing '-' in file name: " + fileName);
            }
            try {
                this.timeStamp = new SimpleDateFormat(TIME_STAMP_FORMAT).parse(fileNameParts[0]);
            }
            catch (ParseException parseException) {
                throw new IllegalArgumentException("Invalid time stamp part of file name: " + fileName);
            }
        }

        Date getTimeStamp() {
            return this.timeStamp;
        }

        void touch(File cacheFolder, ITimeProvider timeProvider) {
            String oldFileName = this.fileName;
            this.createTimeStampAndFileName(timeProvider);
            this.rename(cacheFolder, oldFileName, CacheManager.KEY_FILE_TYPE);
            this.rename(cacheFolder, oldFileName, CacheManager.DATA_FILE_TYPE);
        }

        protected void rename(File cacheFolder, String oldFileName, String fileType) {
            new File(cacheFolder, String.valueOf(oldFileName) + fileType).renameTo(new File(cacheFolder, String.valueOf(this.fileName) + fileType));
        }

        private void createTimeStampAndFileName(ITimeProvider timeProvider) {
            this.timeStamp = new Date(timeProvider.getTimeInMilliseconds());
            this.fileName = new MessageFormat(FILE_NAME_FORMAT).format(new Object[]{this.timeStamp, this.counter++});
        }

        public String toString() {
            return this.fileName;
        }
    }
}

