/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.common.filesystem;

import ch.rinn.restrictions.Private;
import ch.systemsx.cisd.common.exceptions.StatusWithResult;
import ch.systemsx.cisd.common.filesystem.ILastModificationChecker;
import ch.systemsx.cisd.common.filesystem.IStoreItemFilter;
import ch.systemsx.cisd.common.filesystem.StoreItem;
import ch.systemsx.cisd.common.logging.ConditionalNotificationLogger;
import ch.systemsx.cisd.common.logging.LogCategory;
import ch.systemsx.cisd.common.logging.LogFactory;
import ch.systemsx.cisd.common.logging.LogLevel;
import ch.systemsx.cisd.common.utilities.ITimeProvider;
import ch.systemsx.cisd.common.utilities.SystemTimeProvider;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

public class QuietPeriodFileFilter
implements IStoreItemFilter {
    @Private
    static final int CLEANUP_TO_QUIET_PERIOD_RATIO = 10;
    @Private
    static final int MAX_CALLS_BEFORE_CLEANUP = 10000;
    private static final String TIME_FORMAT_TEMPLATE = "%#$tY-%#$tm-%#$td %#$tH:%#$tM:%#$tS";
    private static final Logger operationLog = LogFactory.getLogger(LogCategory.OPERATION, QuietPeriodFileFilter.class);
    private static final Logger machineLog = LogFactory.getLogger(LogCategory.MACHINE, QuietPeriodFileFilter.class);
    private static final Logger notificationLog = LogFactory.getLogger(LogCategory.NOTIFY, QuietPeriodFileFilter.class);
    private final ConditionalNotificationLogger conditionalNotificationLog;
    private final long quietPeriodMillis;
    private final long cleanUpPeriodMillis;
    private final ILastModificationChecker fileStore;
    private final Map<StoreItem, PathCheckRecord> pathMap;
    private final ITimeProvider timeProvider;
    private int callCounter;

    public QuietPeriodFileFilter(ILastModificationChecker fileStore, long quietPeriodMillis, int ignoredErrorCountBeforeNotification) {
        this(fileStore, quietPeriodMillis, SystemTimeProvider.SYSTEM_TIME_PROVIDER, ignoredErrorCountBeforeNotification);
    }

    public QuietPeriodFileFilter(ILastModificationChecker fileStore, long quietPeriodMillis, ITimeProvider timeProvider, int ignoredErrorCountBeforeNotification) {
        assert (quietPeriodMillis > 0L);
        assert (fileStore != null);
        assert (timeProvider != null);
        this.quietPeriodMillis = quietPeriodMillis;
        this.cleanUpPeriodMillis = 10L * quietPeriodMillis;
        this.fileStore = fileStore;
        this.timeProvider = timeProvider;
        this.pathMap = new HashMap<StoreItem, PathCheckRecord>();
        this.callCounter = 0;
        this.conditionalNotificationLog = new ConditionalNotificationLogger(machineLog, notificationLog, ignoredErrorCountBeforeNotification);
    }

    @Override
    public boolean accept(StoreItem item) {
        long now = this.timeProvider.getTimeInMilliseconds();
        try {
            PathCheckRecord checkRecordOrNull = this.pathMap.get(item);
            if (checkRecordOrNull == null) {
                this.saveFirstModificationTime(item, now);
                return false;
            }
            if (now - checkRecordOrNull.getTimeChecked() < this.quietPeriodMillis) {
                return false;
            }
            long oldLastChanged = checkRecordOrNull.getTimeOfLastModification();
            StatusWithResult<Long> status = this.fileStore.lastChanged(item, oldLastChanged);
            if (status.isError()) {
                this.conditionalNotificationLog.log(LogLevel.ERROR, String.format("Cannot obtain \"last changed\" status of item '%s' in store '%s': %s", item, this.fileStore, status));
                return false;
            }
            this.conditionalNotificationLog.reset(null);
            long newLastChanged = status.tryGetResult();
            if (newLastChanged != oldLastChanged) {
                this.pathMap.put(item, new PathCheckRecord(now, newLastChanged));
                if (newLastChanged < oldLastChanged) {
                    machineLog.warn(QuietPeriodFileFilter.getClockProblemLogMessage(item.getName(), checkRecordOrNull.getTimeChecked(), oldLastChanged, now, newLastChanged));
                }
                return false;
            }
            this.pathMap.remove(item);
            return true;
        }
        finally {
            if (++this.callCounter >= 10000) {
                this.cleanUpVanishedPaths(now);
                this.callCounter = 0;
            }
        }
    }

    private void saveFirstModificationTime(StoreItem item, long now) {
        StatusWithResult<Long> status = this.fileStore.lastChanged(item, 0L);
        if (!status.isError()) {
            this.pathMap.put(item, new PathCheckRecord(now, status.tryGetResult()));
            this.conditionalNotificationLog.reset(null);
        } else {
            this.conditionalNotificationLog.log(LogLevel.ERROR, String.format("Cannot obtain \"last changed\" status of item '%s' in store '%s': %s", item, this.fileStore, status));
        }
    }

    private void cleanUpVanishedPaths(long now) {
        Iterator<Map.Entry<StoreItem, PathCheckRecord>> iter = this.pathMap.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<StoreItem, PathCheckRecord> entry = iter.next();
            long timeSinceLastCheckMillis = now - entry.getValue().getTimeChecked();
            if (timeSinceLastCheckMillis <= this.cleanUpPeriodMillis) continue;
            StoreItem item = entry.getKey();
            operationLog.warn("Path '" + item.getName() + "' hasn't been checked for " + (double)timeSinceLastCheckMillis / 1000.0 + " s, removing from path map.");
            iter.remove();
        }
    }

    @Private
    static String getClockProblemLogMessage(String pathname, long oldCheck, long oldLastChanged, long newCheck, long newLastChanged) {
        return String.format("Last modification time of path '%1$s' jumped back: check at " + QuietPeriodFileFilter.getTimeFormat(2) + " -> last modification time " + QuietPeriodFileFilter.getTimeFormat(3) + ", check at later time " + QuietPeriodFileFilter.getTimeFormat(4) + " -> last modification time " + QuietPeriodFileFilter.getTimeFormat(5) + " (which is %6$d ms younger)", pathname, oldCheck, oldLastChanged, newCheck, newLastChanged, oldLastChanged - newLastChanged);
    }

    @Private
    static String getTimeFormat(int position) {
        return StringUtils.replace(TIME_FORMAT_TEMPLATE, "#", Integer.toString(position));
    }

    public static final class PathCheckRecord {
        private final long timeChecked;
        private final long timeOfLastModification;

        public PathCheckRecord(long timeChecked, long timeLastChanged) {
            this.timeChecked = timeChecked;
            this.timeOfLastModification = timeLastChanged;
        }

        public long getTimeChecked() {
            return this.timeChecked;
        }

        public long getTimeOfLastModification() {
            return this.timeOfLastModification;
        }
    }
}

