/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.openbis.generic.server.task;

import ch.rinn.restrictions.Private;
import ch.systemsx.cisd.common.exceptions.ConfigurationFailureException;
import ch.systemsx.cisd.common.exceptions.EnvironmentFailureException;
import ch.systemsx.cisd.common.filesystem.FileUtilities;
import ch.systemsx.cisd.common.logging.LogCategory;
import ch.systemsx.cisd.common.logging.LogFactory;
import ch.systemsx.cisd.common.maintenance.IMaintenanceTask;
import ch.systemsx.cisd.common.properties.PropertyUtils;
import ch.systemsx.cisd.common.utilities.ITimeProvider;
import ch.systemsx.cisd.common.utilities.SystemTimeProvider;
import ch.systemsx.cisd.dbmigration.SimpleDatabaseConfigurationContext;
import ch.systemsx.cisd.openbis.generic.server.CommonServiceProvider;
import ch.systemsx.cisd.openbis.generic.server.ICommonServerForInternalUse;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.CompareType;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.DataTypeCode;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.DetailedSearchCriteria;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.DetailedSearchCriterion;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.DetailedSearchField;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.IEntityProperty;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.Material;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.MaterialAttributeSearchFieldKind;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.MaterialType;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.MaterialTypePropertyType;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.PropertyType;
import ch.systemsx.cisd.openbis.generic.shared.basic.dto.SearchCriteriaConnection;
import ch.systemsx.cisd.openbis.generic.shared.dto.SessionContextDTO;
import ch.systemsx.cisd.openbis.generic.shared.util.DataTypeUtils;
import ch.systemsx.cisd.openbis.generic.shared.util.SimplePropertyValidator;
import java.io.File;
import java.io.Serializable;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import javax.sql.DataSource;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.log4j.Logger;
import org.springframework.jdbc.core.BatchPreparedStatementSetter;
import org.springframework.jdbc.core.ColumnMapRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.support.DatabaseMetaDataCallback;
import org.springframework.jdbc.support.JdbcUtils;
import org.springframework.jdbc.support.MetaDataAccessException;

public class MaterialExternalDBSyncTask
implements IMaintenanceTask {
    @Private
    static final String READ_TIMESTAMP_SQL_KEY = "read-timestamp-sql";
    @Private
    static final String UPDATE_TIMESTAMP_SQL_KEY = "update-timestamp-sql";
    @Private
    static final String INSERT_TIMESTAMP_SQL_KEY = "insert-timestamp-sql";
    @Private
    static final String MAPPING_FILE_KEY = "mapping-file";
    private static final Logger operationLog = LogFactory.getLogger((LogCategory)LogCategory.OPERATION, MaterialExternalDBSyncTask.class);
    private final ICommonServerForInternalUse server;
    private final ITimeProvider timeProvider;
    private Map<String, MappingInfo> mapping;
    private SimpleDatabaseConfigurationContext dbConfigurationContext;
    private JdbcTemplate jdbcTemplate;
    private String readTimestampSql;
    private String insertTimestampSql;
    private String updateTimestampSql;

    public MaterialExternalDBSyncTask() {
        this(CommonServiceProvider.getCommonServer(), (ITimeProvider)SystemTimeProvider.SYSTEM_TIME_PROVIDER);
    }

    public MaterialExternalDBSyncTask(ICommonServerForInternalUse server, ITimeProvider timeProvider) {
        this.server = server;
        this.timeProvider = timeProvider;
    }

    public void setUp(String pluginName, Properties properties) {
        this.dbConfigurationContext = new SimpleDatabaseConfigurationContext(properties);
        this.readTimestampSql = PropertyUtils.getMandatoryProperty((Properties)properties, (String)READ_TIMESTAMP_SQL_KEY);
        this.updateTimestampSql = PropertyUtils.getMandatoryProperty((Properties)properties, (String)UPDATE_TIMESTAMP_SQL_KEY);
        this.insertTimestampSql = properties.getProperty(INSERT_TIMESTAMP_SQL_KEY, this.updateTimestampSql);
        String mappingFileName = PropertyUtils.getMandatoryProperty((Properties)properties, (String)MAPPING_FILE_KEY);
        this.mapping = MaterialExternalDBSyncTask.readMappingFile(mappingFileName);
        Map<String, Map<String, PropertyType>> materialTypes = this.getMaterialTypes();
        Map<String, Map<String, DataTypeCode>> metaData = this.retrieveDatabaseMetaData();
        for (MappingInfo mappingInfo : this.mapping.values()) {
            String materialTypeCode = mappingInfo.getMaterialTypeCode();
            Map<String, PropertyType> propertyTypes = materialTypes.get(materialTypeCode);
            if (propertyTypes == null) {
                throw new ConfigurationFailureException("Mapping file refers to an unknown material type: " + materialTypeCode);
            }
            String tableName = mappingInfo.getTableName();
            Map<String, DataTypeCode> columns = metaData.get(tableName);
            if (columns == null) {
                throw new EnvironmentFailureException("Missing table '" + tableName + "' in report database.");
            }
            mappingInfo.injectDataTypeCodes(columns, propertyTypes);
        }
        this.jdbcTemplate = new JdbcTemplate(this.dbConfigurationContext.getDataSource());
        this.checkTimestampReadingWriting();
    }

    public void execute() {
        SessionContextDTO contextOrNull = this.server.tryToAuthenticateAsSystem();
        if (contextOrNull == null) {
            return;
        }
        Date newTimestamp = new Date(this.timeProvider.getTimeInMilliseconds());
        operationLog.info((Object)"Start reporting added or changed materials to the report database.");
        Map<String, List<Material>> materialsByType = this.getRecentlyAddedOrChangedMaterials(contextOrNull.getSessionToken());
        Set<Map.Entry<String, MappingInfo>> entrySet = this.mapping.entrySet();
        for (Map.Entry<String, MappingInfo> entry : entrySet) {
            String materialTypeCode = entry.getKey();
            List<Material> materials = materialsByType.get(materialTypeCode);
            if (materials == null) continue;
            this.addOrUpdate(entry.getValue(), materials);
        }
        this.writeTimestamp(newTimestamp);
        operationLog.info((Object)"Reporting finished.");
    }

    private Map<String, Map<String, PropertyType>> getMaterialTypes() {
        SessionContextDTO contextOrNull = this.server.tryToAuthenticateAsSystem();
        if (contextOrNull == null) {
            throw new EnvironmentFailureException("Can not authenticate as system.");
        }
        List<MaterialType> materialTypes = this.server.listMaterialTypes(contextOrNull.getSessionToken());
        HashMap<String, Map<String, PropertyType>> result = new HashMap<String, Map<String, PropertyType>>();
        for (MaterialType materialType : materialTypes) {
            List<MaterialTypePropertyType> assignedPropertyTypes = materialType.getAssignedPropertyTypes();
            HashMap<String, PropertyType> propertyTypes = new HashMap<String, PropertyType>();
            for (MaterialTypePropertyType materialTypePropertyType : assignedPropertyTypes) {
                PropertyType propertyType = materialTypePropertyType.getPropertyType();
                propertyTypes.put(propertyType.getCode(), propertyType);
            }
            result.put(materialType.getCode(), propertyTypes);
        }
        return result;
    }

    private Map<String, Map<String, DataTypeCode>> retrieveDatabaseMetaData() {
        Collection<MappingInfo> values = this.mapping.values();
        final HashSet<String> tableNames = new HashSet<String>();
        for (MappingInfo mappingInfo : values) {
            tableNames.add(mappingInfo.getTableName());
        }
        try {
            final HashMap<String, Map<String, DataTypeCode>> map = new HashMap<String, Map<String, DataTypeCode>>();
            JdbcUtils.extractDatabaseMetaData((DataSource)this.dbConfigurationContext.getDataSource(), (DatabaseMetaDataCallback)new DatabaseMetaDataCallback(){

                public Object processMetaData(DatabaseMetaData metaData) throws SQLException, MetaDataAccessException {
                    ResultSet rs = metaData.getColumns(null, null, null, null);
                    while (rs.next()) {
                        String tableName = rs.getString("TABLE_NAME").toLowerCase();
                        if (!tableNames.contains(tableName)) continue;
                        TreeMap<String, DataTypeCode> columns = (TreeMap<String, DataTypeCode>)map.get(tableName);
                        if (columns == null) {
                            columns = new TreeMap<String, DataTypeCode>();
                            map.put(tableName, columns);
                        }
                        String columnName = rs.getString("COLUMN_NAME").toLowerCase();
                        int sqlTypeCode = rs.getInt("DATA_TYPE");
                        DataTypeCode dataTypeCode = DataTypeUtils.getDataTypeCode(sqlTypeCode);
                        columns.put(columnName, dataTypeCode);
                    }
                    rs.close();
                    return null;
                }
            });
            return map;
        }
        catch (MetaDataAccessException ex) {
            throw new ConfigurationFailureException("Couldn't retrieve meta data of database.", (Throwable)ex);
        }
    }

    private void addOrUpdate(MappingInfo mappingInfo, List<Material> materials) {
        List<String> sql = mappingInfo.createSelectStatement(materials);
        List<Map<String, Object>> rows = this.retrieveRowsToBeUpdated(sql);
        Map<String, Map<String, Object>> reportedMaterials = mappingInfo.groupByMaterials(rows);
        ArrayList<Material> newMaterials = new ArrayList<Material>();
        ArrayList<Material> updateMaterials = new ArrayList<Material>();
        for (Material material : materials) {
            Map<String, Object> reportedMaterial = reportedMaterials.get(material.getCode());
            if (reportedMaterial != null) {
                updateMaterials.add(material);
                continue;
            }
            newMaterials.add(material);
        }
        if (!updateMaterials.isEmpty() && mappingInfo.hasProperties()) {
            String updateStatement = mappingInfo.createUpdateStatement();
            this.jdbcTemplate.batchUpdate(updateStatement, mappingInfo.createSetter(updateMaterials, IndexingSchema.UPDATE));
            operationLog.info((Object)(updateMaterials.size() + " materials of type " + mappingInfo.getMaterialTypeCode() + " have been updated in report database."));
        }
        if (!newMaterials.isEmpty()) {
            String insertStatement = mappingInfo.createInsertStatement();
            this.jdbcTemplate.batchUpdate(insertStatement, mappingInfo.createSetter(newMaterials, IndexingSchema.INSERT));
            operationLog.info((Object)(newMaterials.size() + " materials of type " + mappingInfo.getMaterialTypeCode() + " have been inserted into report database."));
        }
    }

    private List<Map<String, Object>> retrieveRowsToBeUpdated(List<String> sqls) {
        ArrayList<Map<String, Object>> rows = new ArrayList<Map<String, Object>>();
        for (String sql : sqls) {
            rows.addAll(this.jdbcTemplate.query(sql, (RowMapper)new ColumnMapRowMapper(){

                protected String getColumnKey(String columnName) {
                    return columnName.toLowerCase();
                }
            }));
        }
        return rows;
    }

    private Map<String, List<Material>> getRecentlyAddedOrChangedMaterials(String sessionToken) {
        DetailedSearchCriteria criteria = new DetailedSearchCriteria();
        DetailedSearchCriterion criterion = new DetailedSearchCriterion(DetailedSearchField.createAttributeField(MaterialAttributeSearchFieldKind.MODIFICATION_DATE), CompareType.MORE_THAN_OR_EQUAL, this.readTimestamp());
        criteria.setCriteria(Arrays.asList(criterion));
        criteria.setConnection(SearchCriteriaConnection.MATCH_ALL);
        List<Material> materials = this.server.searchForMaterials(sessionToken, criteria);
        TreeMap<String, List<Material>> result = new TreeMap<String, List<Material>>();
        for (Material material : materials) {
            String typeCode = material.getMaterialType().getCode();
            ArrayList<Material> list = (ArrayList<Material>)result.get(typeCode);
            if (list == null) {
                list = new ArrayList<Material>();
                result.put(typeCode, list);
            }
            list.add(material);
        }
        return result;
    }

    private void checkTimestampReadingWriting() {
        Date timestamp;
        try {
            timestamp = this.tryToReadTimestamp();
        }
        catch (Exception ex) {
            throw new ConfigurationFailureException("Couldn't get timestamp from report database. Property 'read-timestamp-sql' could be invalid.", (Throwable)ex);
        }
        try {
            this.writeTimestamp(timestamp == null ? new Date(0L) : timestamp);
        }
        catch (Exception ex) {
            throw new ConfigurationFailureException("Couldn't save timestamp to report database. Property 'insert-timestamp-sql' or 'update-timestamp-sql' could be invalid.", (Throwable)ex);
        }
        try {
            this.writeTimestamp(timestamp == null ? new Date(0L) : timestamp);
        }
        catch (Exception ex) {
            throw new ConfigurationFailureException("Couldn't save timestamp to report database. Property 'update-timestamp-sql' could be invalid.", (Throwable)ex);
        }
    }

    private String readTimestamp() {
        Date timestamp = this.tryToReadTimestamp();
        return timestamp == null ? "1970-01-01" : DateFormatUtils.format((Date)timestamp, (String)SimplePropertyValidator.SupportedDatePattern.CANONICAL_DATE_PATTERN.getPattern());
    }

    private Date tryToReadTimestamp() {
        List list = this.jdbcTemplate.queryForList(this.readTimestampSql, Date.class);
        return list.isEmpty() ? null : (Date)list.get(0);
    }

    private void writeTimestamp(Date newTimestamp) {
        String sql = this.tryToReadTimestamp() == null ? this.insertTimestampSql : this.updateTimestampSql;
        this.jdbcTemplate.update(sql, new Object[]{newTimestamp}, new int[]{93});
    }

    @Private
    static Map<String, MappingInfo> readMappingFile(String mappingFileName) {
        LinkedHashMap<String, MappingInfo> map = new LinkedHashMap<String, MappingInfo>();
        List lines = FileUtilities.loadToStringList((File)new File(mappingFileName));
        MappingInfo currentMappingInfo = null;
        for (int i = 0; i < lines.size(); ++i) {
            String[] splittedLine;
            String line = (String)lines.get(i);
            ExecptionFactory factory = new ExecptionFactory(mappingFileName, line, i);
            if (line.startsWith("#") || line.trim().length() == 0) continue;
            if (line.startsWith("[")) {
                if (!line.endsWith("]")) {
                    throw factory.exception("Missing ']'");
                }
                splittedLine = MaterialExternalDBSyncTask.splitAndCheck(line.substring(0, line.length() - 1).substring(1), ":", 2, factory);
                String materialTypeCode = MaterialExternalDBSyncTask.trimAndCheck(splittedLine[0], factory, "material type code");
                splittedLine = MaterialExternalDBSyncTask.splitAndCheck(splittedLine[1], ",", 2, factory);
                String tableName = MaterialExternalDBSyncTask.trimAndCheck(splittedLine[0], factory, "table name");
                String codeColumnName = MaterialExternalDBSyncTask.trimAndCheck(splittedLine[1], factory, "code column name");
                currentMappingInfo = new MappingInfo(materialTypeCode, tableName.toLowerCase(), codeColumnName.toLowerCase());
                map.put(materialTypeCode, currentMappingInfo);
                continue;
            }
            if (currentMappingInfo != null) {
                splittedLine = MaterialExternalDBSyncTask.splitAndCheck(line, ":", 2, factory);
                String propertyTypeCode = MaterialExternalDBSyncTask.trimAndCheck(splittedLine[0], factory, "property type code");
                currentMappingInfo.addPropertyMapping(propertyTypeCode, splittedLine[1].trim().toLowerCase());
                continue;
            }
            throw factory.exception("Missing first material type table definition of form '[<material type tode>: <table name>, <code column name>]'");
        }
        return map;
    }

    private static String[] splitAndCheck(String string, String delimiter, int numberOfItems, ExecptionFactory factory) {
        String[] splittedString = string.split(delimiter);
        if (splittedString.length < numberOfItems) {
            throw factory.exception(numberOfItems + " items separated by '" + delimiter + "' expected.");
        }
        return splittedString;
    }

    private static String trimAndCheck(String string, ExecptionFactory factory, String name) {
        String trimmedString = string.trim();
        if (trimmedString.length() == 0) {
            throw factory.exception("Missing " + name + ".");
        }
        return trimmedString;
    }

    @Private
    void closeDatabaseConnections() {
        if (this.dbConfigurationContext != null) {
            this.dbConfigurationContext.closeConnections();
        }
    }

    private static final class ExecptionFactory {
        private final String mappingFileName;
        private final String line;
        private final int lineIndex;

        ExecptionFactory(String mappingFileName, String line, int lineIndex) {
            this.mappingFileName = mappingFileName;
            this.line = line;
            this.lineIndex = lineIndex;
        }

        ConfigurationFailureException exception(String message) {
            return new ConfigurationFailureException("Error in mapping file '" + this.mappingFileName + "' at line " + (this.lineIndex + 1) + " '" + this.line + "': " + message);
        }
    }

    @Private
    static final class MappingInfo {
        private final int SELECT_CHUNK_SIZE = 500;
        private final String materialTypeCode;
        private final String tableName;
        private final String codeColumnName;
        private final Map<String, Column> propertyMapping = new LinkedHashMap<String, Column>();

        MappingInfo(String materialTypeCode, String tableName, String codeColumnName) {
            this.materialTypeCode = materialTypeCode;
            this.tableName = tableName;
            this.codeColumnName = codeColumnName;
        }

        String getMaterialTypeCode() {
            return this.materialTypeCode;
        }

        String getTableName() {
            return this.tableName;
        }

        String getCodeColumnName() {
            return this.codeColumnName;
        }

        boolean hasProperties() {
            return !this.propertyMapping.isEmpty();
        }

        void addPropertyMapping(String propertyTypeCode, String propertyColumnName) {
            Column column = new Column(propertyColumnName, this.propertyMapping.size());
            this.propertyMapping.put(propertyTypeCode, column);
        }

        void injectDataTypeCodes(Map<String, DataTypeCode> columns, Map<String, PropertyType> propertyTypes) {
            DataTypeCode codeColumnType = columns.remove(this.codeColumnName);
            if (codeColumnType == null) {
                throw new EnvironmentFailureException("Missing column '" + this.codeColumnName + "' in table '" + this.tableName + "' of report database.");
            }
            if (!codeColumnType.equals((Object)DataTypeCode.VARCHAR)) {
                throw new EnvironmentFailureException("Column '" + this.codeColumnName + "' of table '" + this.tableName + "' is not of type VARCHAR.");
            }
            for (Map.Entry<String, Column> entry : this.propertyMapping.entrySet()) {
                String propertyTypeCode = entry.getKey();
                PropertyType propertyType = propertyTypes.get(propertyTypeCode);
                if (propertyType == null) {
                    throw new ConfigurationFailureException("Mapping file refers to an unknown property type: " + propertyTypeCode);
                }
                Column column = entry.getValue();
                DataTypeCode dataTypeCode = columns.get(column.name);
                if (dataTypeCode == null) {
                    throw new EnvironmentFailureException("Missing column '" + column.name + "' in table '" + this.tableName + "' of report database.");
                }
                DataTypeCode correspondingType = this.getCorrespondingType(propertyType.getDataType().getCode());
                if (!dataTypeCode.equals((Object)correspondingType)) {
                    throw new EnvironmentFailureException("Column '" + column.name + "' in table '" + this.tableName + "' of report database should be of a type which corresponds to " + correspondingType + ".");
                }
                column.dataTypeCode = dataTypeCode;
            }
        }

        private DataTypeCode getCorrespondingType(DataTypeCode code) {
            switch (code) {
                case INTEGER: 
                case REAL: 
                case TIMESTAMP: {
                    return code;
                }
            }
            return DataTypeCode.VARCHAR;
        }

        Map<String, Map<String, Object>> groupByMaterials(List<Map<String, Object>> rows) {
            HashMap<String, Map<String, Object>> result = new HashMap<String, Map<String, Object>>();
            for (Map<String, Object> map : rows) {
                String materialCode = map.remove(this.codeColumnName).toString();
                result.put(materialCode, map);
            }
            return result;
        }

        List<String> createSelectStatement(List<Material> materials) {
            ArrayList<String> sqls = new ArrayList<String>();
            int startIndex = 0;
            int endIndex = Math.min(500, materials.size());
            while (startIndex != endIndex) {
                sqls.add(this.createPartialSelectStatement(materials, startIndex, endIndex));
                startIndex = endIndex;
                endIndex = Math.min(endIndex + 500, materials.size());
            }
            return sqls;
        }

        String createPartialSelectStatement(List<Material> materials, int startIndex, int endIndex) {
            StringBuilder builder = new StringBuilder("select * from ");
            builder.append(this.tableName).append(" where ");
            builder.append(this.codeColumnName).append(" in ");
            String delim = "(";
            for (int i = startIndex; i < endIndex; ++i) {
                Material material = materials.get(i);
                builder.append(delim).append('\'').append(material.getCode()).append('\'');
                delim = ", ";
            }
            builder.append(")");
            return builder.toString();
        }

        String createInsertStatement() {
            StringBuilder builder = new StringBuilder("insert into ").append(this.tableName);
            builder.append(" (").append(this.codeColumnName);
            for (Column column : this.propertyMapping.values()) {
                builder.append(", ").append(column.name);
            }
            builder.append(") values(?");
            for (int i = 0; i < this.propertyMapping.size(); ++i) {
                builder.append(", ?");
            }
            builder.append(")");
            return builder.toString();
        }

        String createUpdateStatement() {
            StringBuilder builder = new StringBuilder("update ").append(this.tableName);
            String delim = " set ";
            for (Column column : this.propertyMapping.values()) {
                builder.append(delim).append(column.name).append("=?");
                delim = ", ";
            }
            builder.append(" where ").append(this.codeColumnName).append("=?");
            return builder.toString();
        }

        BatchPreparedStatementSetter createSetter(final List<Material> materials, final IndexingSchema indexing) {
            return new BatchPreparedStatementSetter(){

                public void setValues(PreparedStatement ps, int index) throws SQLException {
                    Material material = (Material)materials.get(index);
                    List<IEntityProperty> properties = material.getProperties();
                    ps.setObject(indexing.getCodeIndex(propertyMapping.size()), material.getCode());
                    int propertyIndexOffset = indexing.getPropertyIndexOffset();
                    for (int i = 0; i < propertyMapping.size(); ++i) {
                        ps.setObject(i + propertyIndexOffset, null);
                    }
                    for (IEntityProperty property : properties) {
                        PropertyType propertyType = property.getPropertyType();
                        String code = propertyType.getCode();
                        Column column = (Column)propertyMapping.get(code);
                        if (column == null) continue;
                        String value = this.getValue(property);
                        Serializable typedValue = DataTypeUtils.convertValueTo(column.dataTypeCode, value);
                        if (typedValue instanceof Date) {
                            ps.setObject(column.index + propertyIndexOffset, (Object)typedValue, 93);
                            continue;
                        }
                        ps.setObject(column.index + propertyIndexOffset, typedValue);
                    }
                }

                public int getBatchSize() {
                    return materials.size();
                }
            };
        }

        private String getValue(IEntityProperty property) {
            String value;
            switch (property.getPropertyType().getDataType().getCode()) {
                case CONTROLLEDVOCABULARY: {
                    value = property.getVocabularyTerm().getCodeOrLabel();
                    break;
                }
                case MATERIAL: {
                    value = property.getMaterial().getCode();
                    break;
                }
                default: {
                    value = property.getValue();
                }
            }
            return value;
        }
    }

    private static enum IndexingSchema {
        INSERT(0, 2),
        UPDATE(-1, 1);

        private final int codeIndex;
        private final int offset;

        private IndexingSchema(int codeIndex, int offset) {
            this.codeIndex = codeIndex;
            this.offset = offset;
        }

        public int getCodeIndex(int size) {
            return 1 + (size + 1 + this.codeIndex) % (size + 1);
        }

        public int getPropertyIndexOffset() {
            return this.offset;
        }
    }

    private static final class Column {
        private final String name;
        private final int index;
        private DataTypeCode dataTypeCode;

        Column(String name, int index) {
            this.name = name;
            this.index = index;
        }
    }
}

