/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.common.maintenance;

import ch.systemsx.cisd.base.exceptions.CheckedExceptionTunnel;
import ch.systemsx.cisd.common.concurrent.ConcurrencyUtilities;
import ch.systemsx.cisd.common.exceptions.ConfigurationFailureException;
import ch.systemsx.cisd.common.filesystem.FileUtilities;
import ch.systemsx.cisd.common.logging.LogCategory;
import ch.systemsx.cisd.common.logging.LogFactory;
import ch.systemsx.cisd.common.maintenance.IDataStoreLockingMaintenanceTask;
import ch.systemsx.cisd.common.maintenance.IMaintenanceTask;
import ch.systemsx.cisd.common.maintenance.INextTimestampProvider;
import ch.systemsx.cisd.common.maintenance.MaintenanceTaskParameters;
import ch.systemsx.cisd.common.reflection.ClassUtils;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.log4j.Logger;

public class MaintenancePlugin {
    static final String TIME_STAMP_FORMAT = "yyyy-MM-dd HH:mm:ss";
    private static final Logger operationLog = LogFactory.getLogger((LogCategory)LogCategory.OPERATION, MaintenancePlugin.class);
    private static ReentrantLock dataStoreLock = new ReentrantLock();
    private final IMaintenanceTask task;
    private final MaintenanceTaskParameters parameters;
    private final boolean requiresDataStoreLock;
    private Timer workerTimer;
    private volatile boolean stopped;

    public MaintenancePlugin(MaintenanceTaskParameters parameters) {
        this.parameters = parameters;
        try {
            this.task = (IMaintenanceTask)ClassUtils.create(IMaintenanceTask.class, (String)parameters.getClassName(), (Object[])new Object[0]);
        }
        catch (Exception ex) {
            throw new ConfigurationFailureException("Cannot find the plugin class '" + parameters.getClassName() + "'", (Throwable)CheckedExceptionTunnel.unwrapIfNecessary((Exception)ex));
        }
        try {
            this.task.setUp(parameters.getPluginName(), parameters.getProperties());
        }
        catch (Throwable t) {
            throw new ConfigurationFailureException("Set up of maintenance task '" + parameters.getPluginName() + "' failed: " + t.getMessage(), t);
        }
        this.requiresDataStoreLock = this.requiresDataStoreLock();
    }

    public MaintenancePlugin(IMaintenanceTask task, MaintenanceTaskParameters parameters) {
        this.parameters = parameters;
        this.task = task;
        this.requiresDataStoreLock = this.requiresDataStoreLock();
    }

    private boolean requiresDataStoreLock() {
        if (this.task instanceof IDataStoreLockingMaintenanceTask) {
            return ((IDataStoreLockingMaintenanceTask)this.task).requiresDataStoreLock();
        }
        return false;
    }

    public String getPluginName() {
        return this.parameters.getPluginName();
    }

    public MaintenanceTaskParameters getParameters() {
        return this.parameters;
    }

    public IMaintenanceTask getTask() {
        return this.task;
    }

    public synchronized void start() {
        String timerThreadName = this.parameters.getPluginName() + " - Maintenance Plugin";
        this.workerTimer = new Timer(timerThreadName);
        MaintenanceTimerTask timerTask = new MaintenanceTimerTask(this.parameters.getRetryIntervals());
        Date startDate = this.parameters.getStartDate();
        INextTimestampProvider nextTimestampProvider = this.parameters.getNextTimestampProvider();
        if (nextTimestampProvider != null) {
            if (this.loadPersistenNextDateOrNull() == null) {
                this.savePersistentNextDate(nextTimestampProvider.getNextTimestamp(new Date()));
            }
            this.schedule(timerTask, nextTimestampProvider);
        } else if (this.parameters.isExecuteOnlyOnce()) {
            this.workerTimer.schedule((TimerTask)timerTask, startDate);
            if (operationLog.isInfoEnabled()) {
                operationLog.info((Object)("Plugin scheduled: " + this.parameters.getPluginName() + ", single execution at " + startDate));
            }
        } else {
            this.workerTimer.schedule((TimerTask)timerTask, startDate, this.parameters.getIntervalSeconds() * 1000L);
            if (operationLog.isInfoEnabled()) {
                operationLog.info((Object)("Plugin scheduled: " + this.parameters.getPluginName() + ", first execution at " + startDate + ", scheduling interval: " + this.parameters.getIntervalSeconds() + "s."));
            }
        }
    }

    private void schedule(final TimerTask timerTask, final INextTimestampProvider nextTimestampProvider) {
        Date timestamp;
        Date savedNext = this.loadPersistenNextDateOrNull();
        Date now = new Date();
        Date next = nextTimestampProvider.getNextTimestamp(now);
        Date date = timestamp = savedNext == null || !next.after(savedNext) ? next : now;
        if (this.workerTimer != null) {
            this.workerTimer.schedule(new TimerTask(){

                @Override
                public void run() {
                    timerTask.run();
                    MaintenancePlugin.this.savePersistentNextDate(nextTimestampProvider.getNextTimestamp(timestamp));
                    MaintenancePlugin.this.schedule(timerTask, nextTimestampProvider);
                }
            }, timestamp);
        }
    }

    private Date loadPersistenNextDateOrNull() {
        File persistentNextDateFile = this.parameters.getPersistentNextDateFile();
        if (persistentNextDateFile.isFile()) {
            String timeStampString = FileUtilities.loadToString((File)persistentNextDateFile).trim();
            try {
                return new SimpleDateFormat(TIME_STAMP_FORMAT).parse(timeStampString);
            }
            catch (ParseException ex) {
                operationLog.warn((Object)("Invalid time stamp in '" + persistentNextDateFile.getAbsolutePath() + "': " + timeStampString));
            }
        }
        return null;
    }

    private void savePersistentNextDate(Date next) {
        File persistentNextDateFile = this.parameters.getPersistentNextDateFile();
        persistentNextDateFile.getParentFile().mkdirs();
        FileUtilities.writeToFile((File)persistentNextDateFile, (String)new SimpleDateFormat(TIME_STAMP_FORMAT).format(next));
    }

    public synchronized void execute() {
        MaintenanceTimerTask timerTask = new MaintenanceTimerTask(this.parameters.getRetryIntervals());
        timerTask.doRun();
    }

    public synchronized void shutdown() {
        if (this.workerTimer != null) {
            this.workerTimer.cancel();
            this.workerTimer = null;
        }
        this.stopped = true;
        if (this.task instanceof Closeable) {
            try {
                ((Closeable)((Object)this.task)).close();
            }
            catch (IOException ex) {
                throw CheckedExceptionTunnel.wrapIfNecessary((Exception)ex);
            }
        }
    }

    private class MaintenanceTimerTask
    extends TimerTask {
        private List<Long> retryIntervals;

        public MaintenanceTimerTask(List<Long> retryIntervals) {
            this.retryIntervals = retryIntervals;
        }

        @Override
        public void run() {
            if (MaintenancePlugin.this.stopped) {
                return;
            }
            this.doRun();
        }

        private void doRun() {
            String className = MaintenancePlugin.this.task.getClass().getCanonicalName();
            int retryCounter = 1;
            while (true) {
                try {
                    this.acquireLockIfNecessary();
                    try {
                        MaintenancePlugin.this.task.execute();
                    }
                    finally {
                        this.releaseLockIfNecessay();
                    }
                    return;
                }
                catch (Throwable th) {
                    try {
                        if (retryCounter <= this.retryIntervals.size()) {
                            long retryInterval = this.retryIntervals.get(retryCounter - 1);
                            operationLog.warn((Object)("Execution of maintenance task '" + className + "' failed. " + retryCounter + ". retry in " + retryInterval + " msec."));
                            ++retryCounter;
                            ConcurrencyUtilities.sleep((long)retryInterval);
                            continue;
                        }
                        throw th;
                    }
                    catch (Throwable th2) {
                        operationLog.error((Object)("Exception when running maintenance task '" + className + "'."), th2);
                        return;
                    }
                }
                break;
            }
        }

        private void acquireLockIfNecessary() {
            if (MaintenancePlugin.this.requiresDataStoreLock) {
                dataStoreLock.lock();
            }
        }

        private void releaseLockIfNecessay() {
            if (MaintenancePlugin.this.requiresDataStoreLock) {
                dataStoreLock.unlock();
            }
        }
    }
}

