/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.common.logging;

import ch.systemsx.cisd.common.logging.LogLog;
import ch.systemsx.cisd.common.logging.ext.DailyRollingFileHandler;
import ch.systemsx.cisd.common.logging.ext.LoggingUtils;
import ch.systemsx.cisd.common.logging.ext.NullHandler;
import ch.systemsx.cisd.common.logging.ext.PatternFormatter;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.logging.ConsoleHandler;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;
import java.util.logging.MemoryHandler;
import java.util.logging.SocketHandler;
import java.util.logging.StreamHandler;
import java.util.stream.Collectors;

public class LogInitializer {
    private static final String LOG_DIRECTORY = "etc";
    private static final String LOG_FILENAME = "logging.properties";
    private static final String FILE_URL_PREFIX = "file:";
    private static final String GLOBAL_LEVEL_KEY = ".global.level";
    public static final String GLOBAL_HANDLERS = ".global.handlerAliases";
    public static final String INTERNAL_HANDLERS = ".handlerAliases";
    private static final String LEVEL_SUFFIX = ".level";
    private static boolean initialized;

    private static URL createURL(String configurationOrNull) {
        LogLog.info("[LogInitializer] createURL(config=" + configurationOrNull + ")");
        if (configurationOrNull != null) {
            try {
                LogLog.info("[LogInitializer] Trying URL string: " + configurationOrNull);
                return new URL(configurationOrNull);
            }
            catch (MalformedURLException ex) {
                System.err.println("[LogInitializer] Malformed URL '" + configurationOrNull + "': " + ex.getMessage());
            }
        }
        URL resource = LogInitializer.class.getResource("/etc/logging.properties");
        LogLog.info("[LogInitializer] Classpath resource URL: " + String.valueOf(resource));
        return resource;
    }

    private static String tryFindConfigurationInSystemProperties() {
        String trimmed;
        String config = System.getProperty("log4j.configuration");
        LogLog.info("[LogInitializer] System property log4j.configuration=" + config);
        if (config != null && !(trimmed = config.trim()).isEmpty()) {
            LogLog.info("[LogInitializer] Using system property config: " + trimmed);
            return trimmed;
        }
        return null;
    }

    private static File getEffectiveLogFile(String configurationOrNull) {
        File file = configurationOrNull == null ? new File(LOG_DIRECTORY, LOG_FILENAME) : new File(configurationOrNull);
        LogLog.info("[LogInitializer] getEffectiveLogFile -> " + file.getAbsolutePath());
        return file;
    }

    public static synchronized void init() {
        LogLog.info("[LogInitializer] init() called. alreadyInitialized=" + initialized);
        if (initialized) {
            LogLog.info("[LogInitializer] Already initialized. Skipping.");
            return;
        }
        LogInitializer.initialize();
    }

    private static void initialize() {
        LogLog.info("[LogInitializer] Starting initialize()");
        String config = LogInitializer.tryFindConfigurationInSystemProperties();
        boolean fileTried = false;
        if (config == null || !config.startsWith(FILE_URL_PREFIX)) {
            File logFile = LogInitializer.getEffectiveLogFile(config);
            fileTried = true;
            if (logFile.exists() && logFile.isFile()) {
                LogLog.info("[LogInitializer] Found log file: " + logFile.getAbsolutePath());
                LogInitializer.configureFromFile(logFile);
                initialized = true;
                LogLog.info("[LogInitializer] Initialized from file.");
                return;
            }
            LogLog.info("[LogInitializer] Log file not found: " + logFile.getAbsolutePath());
        }
        LogLog.info("[LogInitializer] Trying URL-based config (config=" + config + ")");
        URL url = LogInitializer.createURL(config);
        if (url != null) {
            LogInitializer.configureFromURL(url);
            initialized = true;
            LogLog.info("[LogInitializer] Initialized from URL.");
            return;
        }
        initialized = true;
        LogLog.info("[LogInitializer] No external config found. Initialization complete with defaults.");
    }

    public static void configureFromURL(URL url) {
        LogLog.info("[LogInitializer] configureFromURL(" + String.valueOf(url) + ")");
        try {
            File file = new File(url.toURI());
            LogLog.info("[LogInitializer] URL to URI path: " + file.getAbsolutePath());
            if (file.exists()) {
                LogLog.info("[LogInitializer] File exists at URL path. Delegating to configureFromFile.");
                LogInitializer.configureFromFile(file);
                return;
            }
        }
        catch (URISyntaxException e) {
            System.err.println("[LogInitializer] URI syntax error: " + e.getMessage());
        }
        System.out.printf("[LogInitializer] Configured from URL '%s' (no watching)%n", url);
    }

    public static void configureFromFile(File logFile) {
        if (!logFile.getName().endsWith(".properties")) {
            String msg = String.format("[LogInitializer] Unsupported configuration file: '%s'.%nPlease supply a '.properties' file for logging setup.", logFile.getAbsolutePath());
            LogLog.info(msg);
            new IllegalArgumentException("Invalid file type").printStackTrace(System.out);
            return;
        }
        LogLog.info("[LogInitializer] configureFromFile(logFile=" + logFile.getAbsolutePath() + ")");
        LogInitializer.configureFromFile(logFile, null);
    }

    public static void configureFromFile(File logFileParam, List<String> loggersToInitialize) {
        Properties props;
        LogLog.info("[LogInitializer] configureFromFile(loggers override) called.");
        File logFile = logFileParam;
        if (logFileParam == null || logFileParam.getPath().trim().isEmpty()) {
            System.err.println("[LogInitializer] No log file defined; using default logging.properties in cwd.");
            logFile = new File(LOG_FILENAME);
        }
        if ((props = LogInitializer.loadProperties(logFile)) == null) {
            System.err.println("[LogInitializer] Properties load failed; aborting.");
            return;
        }
        try (FileInputStream in = new FileInputStream(logFile);){
            LogManager.getLogManager().readConfiguration(in);
            LogLog.info("[LogInitializer][Bootstrap] Loaded JUL config from " + String.valueOf(logFile));
        }
        catch (IOException ioe) {
            LogLog.info("[LogInitializer][Bootstrap] Couldn't to load JUL config from " + String.valueOf(logFile) + ": " + ioe.getMessage());
        }
        String handlersList = props.getProperty(GLOBAL_HANDLERS);
        if (handlersList == null) {
            System.err.println("[LogInitializer] No 'handlers' property defined.");
            LogInitializer.configureGlobalLoggingLevel(props);
            return;
        }
        LogLog.info("[LogInitializer] Handlers to configure: " + handlersList);
        String[] splitHandlerAliases = handlersList.split("\\s*,\\s*");
        Set<String> handlerAliases = Arrays.stream(splitHandlerAliases).map(String::trim).collect(Collectors.toSet());
        LogInitializer.removeAllHandlersBeforeInitialization();
        LogInitializer.configureGlobalLoggingLevel(props);
        for (String alias : handlerAliases) {
            LogLog.info("[LogInitializer] Initializing handler: " + alias);
            try {
                Handler handler = LogInitializer.createHandler(alias, props);
                LogInitializer.configureCommonProperties(handler, alias, props);
                LogInitializer.configureSpecialProperties(handler, alias, props);
                Logger.getLogger("").addHandler(handler);
                LogLog.info("[LogInitializer] Handler added: " + String.valueOf(handler));
            }
            catch (Exception e) {
                System.err.println("[LogInitializer] Error initializing handler '" + alias + "': " + e.getMessage());
                e.printStackTrace();
            }
        }
        for (String key : props.stringPropertyNames()) {
            if (!key.endsWith(INTERNAL_HANDLERS) || key.equals(GLOBAL_HANDLERS)) continue;
            String loggerName = key.substring(0, key.length() - INTERNAL_HANDLERS.length());
            String[] aliases2 = props.getProperty(key).split("\\s*,\\s*");
            Logger logger = null;
            try {
                System.err.println(logFile.getAbsolutePath() + " : " + loggerName);
                logger = Logger.getLogger(loggerName);
            }
            catch (Throwable e) {
                System.out.println(logFile.getAbsolutePath() + String.valueOf(e));
                throw new RuntimeException(logFile.getAbsolutePath(), e);
            }
            for (Handler h : logger.getHandlers()) {
                logger.removeHandler(h);
            }
            for (String alias2 : aliases2) {
                try {
                    Handler handler = LogInitializer.createHandler(alias2, props);
                    LogInitializer.configureCommonProperties(handler, alias2, props);
                    LogInitializer.configureSpecialProperties(handler, alias2, props);
                    logger.addHandler(handler);
                    LogLog.info("[LogInitializer] Attached handler '" + (String)alias2 + "' to logger '" + loggerName + "'");
                }
                catch (Exception e) {
                    System.err.println("[LogInitializer] Failed to attach handler '" + (String)alias2 + "' to logger '" + loggerName + "': " + e.getMessage());
                    e.printStackTrace();
                }
            }
        }
        LogInitializer.configureSpecificLoggerLevels(props, handlerAliases);
    }

    public static Thread watchLoggerLevel(String loggerName, long pollIntervalMillis) {
        Logger logger1 = Logger.getLogger(loggerName);
        Level[] previousLevel = new Level[]{logger1.getLevel()};
        DateTimeFormatter fmt = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        Thread watcher = new Thread(() -> {
            while (!Thread.currentThread().isInterrupted()) {
                Logger logger = Logger.getLogger(loggerName);
                Level current = logger.getLevel();
                if (current != previousLevel[0]) {
                    String time = LocalDateTime.now().format(fmt);
                    System.out.printf("[%s] Logger '%s' level changed from %s to %s%n", time, loggerName, previousLevel[0], current);
                    previousLevel[0] = current;
                }
                try {
                    Thread.sleep(pollIntervalMillis);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
            }
        }, "LogLevelWatcher-" + loggerName);
        watcher.setDaemon(true);
        watcher.start();
        return watcher;
    }

    private static void removeAllHandlersBeforeInitialization() {
        LogLog.info("[LogInitializer] removeAllHandlersBeforeInitialization()");
        Logger root = Logger.getLogger("");
        for (Handler h : root.getHandlers()) {
            LogLog.info("[LogInitializer] Removing handler: " + String.valueOf(h));
            root.removeHandler(h);
        }
        LogLog.info("[LogInitializer] also reset : LogManager.getLogManager().reset()");
    }

    private static Properties loadProperties(File configurationFile) {
        LogLog.info("[LogInitializer] loadProperties(file=" + configurationFile.getPath() + ")");
        Properties props = new Properties();
        InputStream is = null;
        boolean loaded = false;
        if (configurationFile.exists() && configurationFile.isFile()) {
            try {
                LogLog.info("[LogInitializer] Loading properties from filesystem: " + configurationFile.getAbsolutePath());
                is = new FileInputStream(configurationFile);
            }
            catch (IOException e) {
                System.err.println("[LogInitializer] IOException opening file: " + e.getMessage());
            }
        } else {
            LogLog.info("[LogInitializer] File not found on filesystem; will try classpath lookup.");
        }
        if (is == null) {
            String resourcePath = configurationFile.getPath().replace(File.separatorChar, '/');
            if (resourcePath.startsWith("/")) {
                resourcePath = resourcePath.substring(1);
            }
            LogLog.info("[LogInitializer] Trying classpath resource: " + resourcePath);
            ClassLoader cl = Thread.currentThread().getContextClassLoader();
            if (cl != null) {
                is = cl.getResourceAsStream(resourcePath);
                LogLog.info("[LogInitializer] ContextClassLoader resourceStream=" + (is != null));
            }
            if (is == null) {
                is = LogInitializer.class.getResourceAsStream("/" + resourcePath);
                LogLog.info("[LogInitializer] Class.getResourceAsStream(/" + resourcePath + ")=" + (is != null));
                if (is == null) {
                    is = LogInitializer.class.getResourceAsStream(resourcePath);
                    LogLog.info("[LogInitializer] Class.getResourceAsStream(" + resourcePath + ")=" + (is != null));
                }
            }
        }
        if (is != null) {
            try (FileInputStream in = is;){
                props.load(in);
                loaded = true;
                LogLog.info("[LogInitializer] Properties loaded successfully.");
            }
            catch (IOException e) {
                System.err.println("[LogInitializer] Error loading properties: " + e.getMessage());
            }
        } else {
            System.err.println("[LogInitializer] Could not find configuration resource: " + configurationFile.getPath());
        }
        return loaded ? props : null;
    }

    private static void configureGlobalLoggingLevel(Properties props) {
        Level lvl;
        LogLog.info("[LogInitializer] configureGlobalLoggingLevel");
        String global = props.getProperty(GLOBAL_LEVEL_KEY);
        LogLog.info("[LogInitializer] .global.level=" + global);
        if (global != null && !global.isEmpty() && (lvl = LogInitializer.tryGetLogLevel(global)) != null) {
            Logger.getLogger("").setLevel(lvl);
            LogLog.info("[LogInitializer] Root logger level set to " + String.valueOf(lvl));
        }
    }

    private static Level tryGetLogLevel(String levelStr) {
        LogLog.info("[LogInitializer] tryGetLogLevel(" + levelStr + ")");
        if (levelStr == null || levelStr.isEmpty()) {
            return null;
        }
        try {
            org.apache.log4j.Level log4jLevel = org.apache.log4j.Level.parse(levelStr);
            Level jul = LoggingUtils.mapToJULLevel(log4jLevel);
            LogLog.info("[LogInitializer] Parsed log4j level " + String.valueOf(log4jLevel) + " -> JUL level " + String.valueOf(jul));
            return jul;
        }
        catch (Exception e) {
            try {
                Level jul = Level.parse(levelStr);
                LogLog.info("[LogInitializer] Parsed JUL level " + String.valueOf(jul));
                return jul;
            }
            catch (Exception ex) {
                System.err.println("[LogInitializer] Failed to parse level: " + levelStr);
                return null;
            }
        }
    }

    private static Handler createHandler(String alias, Properties props) throws Exception {
        Handler handler;
        LogLog.info("[LogInitializer] createHandler(alias=" + alias + ")");
        String className = props.getProperty(alias + ".class");
        LogLog.info("[LogInitializer] Handler class name=" + className);
        if (className == null) {
            throw new IllegalArgumentException("No class defined for handler alias: \"" + alias + "\"");
        }
        Class<?> hc = Class.forName(className);
        if (DailyRollingFileHandler.class.isAssignableFrom(hc)) {
            String pattern = props.getProperty(alias + ".logFileNamePattern");
            int limit = Integer.parseInt(props.getProperty(alias + ".maxLogFileSize", "0"));
            boolean append = Boolean.parseBoolean(props.getProperty(alias + ".append", "false"));
            LogLog.info("[LogInitializer] Using file-handler constructor: pattern=" + pattern + ", limit=" + limit + ", append=" + append);
            handler = (Handler)hc.getConstructor(String.class, Integer.TYPE, Boolean.TYPE).newInstance(pattern, limit, append);
        } else if (ConsoleHandler.class.isAssignableFrom(hc) || NullHandler.class.isAssignableFrom(hc) || Handler.class.isAssignableFrom(hc)) {
            LogLog.info("[LogInitializer] Using no-arg constructor for handler");
            handler = (Handler)hc.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
        } else {
            throw new IllegalArgumentException("Handler not found : " + String.valueOf(hc));
        }
        return handler;
    }

    private static void configureCommonProperties(Handler handler, String alias, Properties props) throws Exception {
        String formatterClassName;
        String levelStr = props.getProperty(alias + LEVEL_SUFFIX);
        if (levelStr != null) {
            handler.setLevel(LogInitializer.tryGetLogLevel(levelStr));
        }
        if ((formatterClassName = props.getProperty(alias + ".formatter")) != null) {
            Class<?> formatterClass = Class.forName(formatterClassName);
            Formatter formatter = (Formatter)formatterClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            handler.setFormatter(formatter);
        } else {
            String messagePattern = props.getProperty(alias + ".messagePattern");
            if (messagePattern != null) {
                PatternFormatter patternFormatter = new PatternFormatter(messagePattern);
                handler.setFormatter(patternFormatter);
            }
        }
        String encoding = props.getProperty(alias + ".encoding");
        if (encoding != null) {
            handler.setEncoding(encoding);
        }
    }

    private static void configureSpecialProperties(Handler handler, String alias, Properties props) throws Exception {
        if (handler instanceof SocketHandler) {
            LogInitializer.configureSocketHandler((SocketHandler)handler, alias, props);
        }
        if (handler instanceof MemoryHandler) {
            LogInitializer.configureMemoryHandler((MemoryHandler)handler, alias, props);
        }
    }

    private static void configureSocketHandler(SocketHandler handler, String alias, Properties props) {
        String portStr;
        String host = props.getProperty(alias + ".host");
        if (host != null) {
            try {
                Method setHost = handler.getClass().getMethod("setHost", String.class);
                setHost.invoke((Object)handler, host);
            }
            catch (NoSuchMethodException nsme) {
                try {
                    Field hostField = handler.getClass().getDeclaredField("host");
                    hostField.setAccessible(true);
                    hostField.set(handler, host);
                }
                catch (Exception ex) {
                    System.err.println("Could not set host for SocketHandler for alias: " + alias);
                }
            }
            catch (Exception e) {
                System.err.println("Error setting host for SocketHandler for alias: " + alias);
            }
        }
        if ((portStr = props.getProperty(alias + ".port")) != null) {
            int port = Integer.parseInt(portStr);
            try {
                Method setPort = handler.getClass().getMethod("setPort", Integer.TYPE);
                setPort.invoke((Object)handler, port);
            }
            catch (NoSuchMethodException nsme) {
                try {
                    Field portField = handler.getClass().getDeclaredField("port");
                    portField.setAccessible(true);
                    portField.set(handler, port);
                }
                catch (Exception ex) {
                    System.err.println("Could not set port for SocketHandler for alias: " + alias);
                }
            }
            catch (Exception e) {
                System.err.println("Error setting port for SocketHandler for alias: " + alias);
            }
        }
    }

    private static void configureMemoryHandler(MemoryHandler handler, String alias, Properties props) {
        String targetClassName;
        String targetAlias;
        String pushLevelStr;
        String sizeStr = props.getProperty(alias + ".size");
        if (sizeStr != null) {
            int size = Integer.parseInt(sizeStr);
            try {
                Field sizeField = handler.getClass().getDeclaredField("size");
                sizeField.setAccessible(true);
                sizeField.set(handler, size);
            }
            catch (Exception e) {
                System.err.println("Could not set size for MemoryHandler for alias: " + alias);
            }
        }
        if ((pushLevelStr = props.getProperty(alias + ".push")) != null) {
            Level pushLevel = LogInitializer.tryGetLogLevel(pushLevelStr);
            try {
                Method setPushLevel = handler.getClass().getMethod("setPushLevel", Level.class);
                setPushLevel.invoke((Object)handler, pushLevel);
            }
            catch (Exception e) {
                System.err.println("Could not set push level for MemoryHandler for alias: " + alias);
            }
        }
        if ((targetAlias = props.getProperty(alias + ".target")) != null && (targetClassName = props.getProperty(targetAlias + ".class")) != null) {
            try {
                Handler targetHandler;
                Class<?> targetClass = Class.forName(targetClassName);
                if (targetClass.equals(ConsoleHandler.class) || targetClass.equals(StreamHandler.class) || targetClass.equals(SocketHandler.class) || targetClass.equals(MemoryHandler.class)) {
                    targetHandler = (Handler)targetClass.getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                } else {
                    String pattern = props.getProperty(targetAlias + ".logFileNamePattern");
                    int limit = Integer.parseInt(props.getProperty(targetAlias + ".maxLogFileSize", "0"));
                    int count = Integer.parseInt(props.getProperty(targetAlias + ".maxBackupFiles", "1"));
                    boolean append = Boolean.parseBoolean(props.getProperty(targetAlias + ".append", "false"));
                    targetHandler = (Handler)targetClass.getConstructor(String.class, Integer.TYPE, Integer.TYPE, Boolean.TYPE).newInstance(pattern, limit, count, append);
                }
                Field targetField = handler.getClass().getDeclaredField("target");
                targetField.setAccessible(true);
                targetField.set(handler, targetHandler);
            }
            catch (Exception e) {
                System.err.println("Could not set target for MemoryHandler for alias: " + alias);
            }
        }
    }

    private static void configureSpecificLoggerLevels(Properties props, Set<String> handlerAliases) {
        LogLog.info("Configuring specific logger levels...");
        for (String key : props.stringPropertyNames()) {
            String namePart;
            if (!key.endsWith(LEVEL_SUFFIX) || key.equals(GLOBAL_LEVEL_KEY) || handlerAliases.contains(namePart = key.substring(0, key.length() - LEVEL_SUFFIX.length()))) continue;
            String loggerName = namePart;
            String levelStr = props.getProperty(key);
            Level level = LogInitializer.tryGetLogLevel(levelStr);
            if (level != null) {
                try {
                    Logger logger = Logger.getLogger(loggerName);
                    logger.setLevel(level);
                    LogLog.info("Set level " + level.getName() + " for logger '" + loggerName + "'");
                }
                catch (Exception e) {
                    System.err.println("ERROR: Could not set level for logger '" + loggerName + "' from property '" + key + "': " + e.getMessage());
                    e.printStackTrace(System.err);
                }
                continue;
            }
            System.err.println("WARNING: Could not parse level '" + levelStr + "' for logger property '" + key + "'. Level not set.");
        }
        LogLog.info("Finished configuring specific logger levels.");
    }

    static {
        System.setProperty("log4j.defaultInitOverride", "true");
        LogLog.info("[LogInitializer] Static init: log4j.defaultInitOverride=true");
        System.setProperty("org.jboss.logging.provider", "jdk");
        initialized = false;
    }
}

