/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.common.logging.ext;

import java.io.File;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.logging.Handler;
import java.util.logging.LogRecord;

public class DailyRollingFileHandler
extends Handler {
    private final int maxLogFileSize;
    private final boolean append;
    private final String logFileNamePattern;
    private File currentFile;
    private FileOutputStream outputStream;
    private final DateTimeFormatter dateFormatter;
    private LocalDate currentDate;

    public DailyRollingFileHandler(String logFileNamePattern) throws IOException {
        this(logFileNamePattern, Integer.MAX_VALUE, true, ".yyyy-MM-dd");
    }

    public DailyRollingFileHandler(String logFileNamePattern, int maxLogFileSize, boolean append) throws IOException {
        this(logFileNamePattern, maxLogFileSize, append, ".yyyy-MM-dd");
    }

    public DailyRollingFileHandler(String logFileNamePattern, int maxLogFileSize, boolean append, String datePattern) throws IOException {
        this.logFileNamePattern = logFileNamePattern;
        this.maxLogFileSize = maxLogFileSize;
        this.append = append;
        this.dateFormatter = DateTimeFormatter.ofPattern(datePattern);
        this.currentDate = LocalDate.now();
        this.openActiveFile();
    }

    private void openActiveFile() throws IOException {
        File parent;
        this.currentFile = new File(this.logFileNamePattern);
        if (!this.append && this.currentFile.exists()) {
            this.currentFile.delete();
        }
        if ((parent = this.currentFile.getParentFile()) != null && !parent.exists()) {
            parent.mkdirs();
        }
        this.outputStream = new FileOutputStream(this.currentFile, this.append);
    }

    private File datedFile(LocalDate date) {
        String suffix = date.format(this.dateFormatter);
        return new File(this.logFileNamePattern + suffix);
    }

    private void rotateByDate() throws IOException {
        this.outputStream.close();
        File renamed = this.datedFile(this.currentDate);
        if (!this.currentFile.renameTo(renamed)) {
            throw new IOException("Failed to rollover " + String.valueOf(this.currentFile) + " to " + String.valueOf(renamed));
        }
        this.openActiveFile();
        this.currentDate = LocalDate.now();
    }

    private void rotateBySize() throws IOException {
        this.outputStream.close();
        LocalDate date = this.currentDate;
        String suffix = date.format(this.dateFormatter);
        File parentDir = this.currentFile.getParentFile() != null ? this.currentFile.getParentFile() : new File(".");
        String prefix = new File(this.logFileNamePattern).getName() + suffix + ".";
        int nextIndex = DailyRollingFileHandler.getNextIndex(parentDir, prefix);
        File rotated = new File(parentDir, new File(this.logFileNamePattern).getName() + suffix + "." + nextIndex);
        if (!this.currentFile.renameTo(rotated)) {
            throw new IOException("Failed to rename " + String.valueOf(this.currentFile) + " to " + String.valueOf(rotated));
        }
        this.openActiveFile();
    }

    private static int getNextIndex(File parentDir, final String prefix) {
        File[] existing = parentDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(prefix) && name.substring(prefix.length()).matches("\\d+");
            }
        });
        int nextIndex = 1;
        if (existing != null && existing.length > 0) {
            for (File f : existing) {
                try {
                    int idx = Integer.parseInt(f.getName().substring(prefix.length()));
                    if (idx < nextIndex) continue;
                    nextIndex = idx + 1;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
        }
        return nextIndex;
    }

    @Override
    public synchronized void publish(LogRecord record) {
        if (!this.isLoggable(record)) {
            return;
        }
        try {
            LocalDate now = LocalDate.now();
            if (!now.equals(this.currentDate)) {
                this.rotateByDate();
            }
            String msg = this.getFormatter().format(record);
            byte[] bytes = msg.getBytes(StandardCharsets.UTF_8);
            if (this.currentFile.length() + (long)bytes.length > (long)this.maxLogFileSize) {
                this.rotateBySize();
            }
            this.outputStream.write(bytes);
            this.outputStream.flush();
        }
        catch (IOException e) {
            this.reportError(null, e, 1);
        }
    }

    @Override
    public void flush() {
        try {
            if (this.outputStream != null) {
                this.outputStream.flush();
            }
        }
        catch (IOException e) {
            this.reportError(null, e, 2);
        }
    }

    @Override
    public void close() throws SecurityException {
        try {
            if (this.outputStream != null) {
                this.outputStream.close();
            }
        }
        catch (IOException e) {
            this.reportError(null, e, 3);
        }
    }

    public String getLogFileNamePattern() {
        return this.logFileNamePattern;
    }

    public int getMaxLogFileSize() {
        return this.maxLogFileSize;
    }

    public boolean isAppend() {
        return this.append;
    }
}

