/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.common.logging.ext;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Handler;
import java.util.logging.LogRecord;

public class DailyRollingFileHandler
extends Handler {
    public static final String DEFAULT_FILE_NAME_DATE_PATTERN = ".yyyy-MM-dd";
    public static final int DEFAULT_MAX_LOG_ROTATIONS = 7;
    public static final int DEFAULT_MAX_LOG_FILE_SIZE = -1;
    public static final Charset DEFAULT_ENCODING = StandardCharsets.UTF_8;
    private final int maxLogFileSize;
    private final boolean append;
    private final String logFileName;
    private final Charset encoding;
    private File currentFile;
    private FileOutputStream outputStream;
    private final DateTimeFormatter dateFormatter;
    private final int maxLogRotations;
    private LocalDate currentDate;
    private long currentSize;
    private final ReentrantLock lock = new ReentrantLock();

    public DailyRollingFileHandler(String logFileName) throws IOException {
        this(logFileName, -1, true, DEFAULT_FILE_NAME_DATE_PATTERN, 7, DEFAULT_ENCODING);
    }

    public DailyRollingFileHandler(String logFileName, int maxLogFileSize, boolean append, int maxLogRotations) throws IOException {
        this(logFileName, maxLogFileSize, append, DEFAULT_FILE_NAME_DATE_PATTERN, maxLogRotations, DEFAULT_ENCODING);
    }

    public DailyRollingFileHandler(String logFileName, int maxLogFileSize, boolean append, String datePattern, int maxLogRotations) throws IOException {
        this(logFileName, maxLogFileSize, append, datePattern, maxLogRotations, DEFAULT_ENCODING);
    }

    public DailyRollingFileHandler(String logFileName, int maxLogFileSize, boolean append, String datePattern, int maxLogRotations, Charset encoding) throws IOException {
        this.logFileName = logFileName;
        this.maxLogFileSize = maxLogFileSize;
        this.append = append;
        this.dateFormatter = DateTimeFormatter.ofPattern(datePattern);
        this.currentDate = LocalDate.now();
        this.maxLogRotations = maxLogRotations;
        this.encoding = encoding;
        this.openActiveFile();
    }

    private void openActiveFile() throws IOException {
        this.currentFile = new File(this.logFileName);
        File parent = this.currentFile.getParentFile();
        if (!(parent == null || parent.exists() || parent.mkdirs() || parent.isDirectory())) {
            throw new IOException("Failed to create parent directories for log file: " + String.valueOf(parent));
        }
        if (!this.append && this.currentFile.exists() && !this.currentFile.delete()) {
            throw new IOException("Failed to delete existing log file: " + String.valueOf(this.currentFile));
        }
        this.outputStream = new FileOutputStream(this.currentFile, this.append);
        this.currentSize = this.currentFile.exists() ? this.currentFile.length() : 0L;
    }

    private File datedFile(LocalDate date) {
        String suffix = date.format(this.dateFormatter);
        return new File(this.logFileName + suffix);
    }

    private void rotateByDate() throws IOException {
        if (this.maxLogRotations == 0) {
            this.truncateActive();
        } else {
            File archived = this.datedFile(this.currentDate);
            this.archiveAndTruncate(archived);
        }
        this.cleanupArchives();
        this.currentDate = LocalDate.now();
    }

    private void rotateBySize() throws IOException {
        if (this.maxLogRotations == 0) {
            this.truncateActive();
        } else {
            LocalDate date = this.currentDate;
            String suffix = date.format(this.dateFormatter);
            File parentDir = this.currentFile.getParentFile() != null ? this.currentFile.getParentFile() : new File(".");
            String prefix = new File(this.logFileName).getName() + suffix + "_";
            int nextIndex = DailyRollingFileHandler.getNextIndex(parentDir, prefix);
            File archive = new File(parentDir, prefix + nextIndex);
            this.archiveAndTruncate(archive);
        }
        this.cleanupArchives();
    }

    private void archiveAndTruncate(File target) throws IOException {
        this.flush();
        DailyRollingFileHandler.copyFile(this.currentFile, target);
        this.truncateActive();
    }

    public static void copyFile(File source, File destination) throws IOException {
        if (!source.exists() || !source.isFile()) {
            throw new FileNotFoundException("Source file not found or not a file: " + source.getAbsolutePath());
        }
        File parentDir = destination.getParentFile();
        if (parentDir != null && !parentDir.exists() && !parentDir.mkdirs()) {
            throw new IOException("Failed to create parent directories for: " + destination.getAbsolutePath());
        }
        try (FileChannel sourceChannel = new FileInputStream(source).getChannel();
             FileChannel destChannel = new FileOutputStream(destination).getChannel();){
            long transferred;
            long size = sourceChannel.size();
            long chunkSize = 0x2000000L;
            for (long position = 0L; position < size; position += transferred) {
                transferred = sourceChannel.transferTo(position, chunkSize, destChannel);
                if (transferred > 0L) continue;
                break;
            }
        }
    }

    private void truncateActive() throws IOException {
        try (RandomAccessFile raf = new RandomAccessFile(this.currentFile, "rw");){
            raf.setLength(0L);
            this.currentSize = 0L;
        }
        catch (IOException e) {
            this.reportError("Failed to truncate log file", e, 1);
            throw e;
        }
    }

    private void cleanupArchives() {
        String baseName;
        if (this.maxLogRotations < 0) {
            return;
        }
        File parentDir = this.currentFile.getParentFile() != null ? this.currentFile.getParentFile() : new File(".");
        File[] archives = parentDir.listFiles((arg_0, arg_1) -> DailyRollingFileHandler.lambda$cleanupArchives$0(baseName = new File(this.logFileName).getName(), arg_0, arg_1));
        if (archives == null || archives.length <= this.maxLogRotations) {
            return;
        }
        Arrays.sort(archives, DailyRollingFileHandler::compareLogFiles);
        int deleteCount = archives.length - this.maxLogRotations;
        for (int i = 0; i < deleteCount; ++i) {
            if (archives[i].delete()) continue;
            this.reportError("Could not delete old log file: " + archives[i].getName(), null, 1);
        }
    }

    private static int compareLogFiles(File f1, File f2) {
        int baseCompare;
        String name1 = f1.getName();
        String name2 = f2.getName();
        int i1_lastUnderscore = name1.lastIndexOf(95);
        int i2_lastUnderscore = name2.lastIndexOf(95);
        String base1 = name1;
        int index1 = -1;
        if (i1_lastUnderscore != -1) {
            try {
                index1 = Integer.parseInt(name1.substring(i1_lastUnderscore + 1));
                base1 = name1.substring(0, i1_lastUnderscore);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        String base2 = name2;
        int index2 = -1;
        if (i2_lastUnderscore != -1) {
            try {
                index2 = Integer.parseInt(name2.substring(i2_lastUnderscore + 1));
                base2 = name2.substring(0, i2_lastUnderscore);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        if ((baseCompare = base1.compareTo(base2)) != 0) {
            return baseCompare;
        }
        return Integer.compare(index1, index2);
    }

    private static int getNextIndex(File parentDir, final String prefix) {
        File[] existing = parentDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(prefix) && name.substring(prefix.length()).matches("\\d+");
            }
        });
        int nextIndex = 1;
        if (existing != null) {
            for (File f : existing) {
                try {
                    int idx = Integer.parseInt(f.getName().substring(prefix.length()));
                    if (idx < nextIndex) continue;
                    nextIndex = idx + 1;
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
        }
        return nextIndex;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void publish(LogRecord record) {
        if (!this.isLoggable(record)) {
            return;
        }
        byte[] bytes = this.getMessageBytes(record);
        if (bytes == null) {
            return;
        }
        this.lock.lock();
        try {
            this.handleLogRotation(bytes);
            this.outputStream.write(bytes);
            this.outputStream.flush();
            this.currentSize += (long)bytes.length;
        }
        catch (IOException e) {
            this.reportError(null, e, 1);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void handleLogRotation(byte[] bytes) throws IOException {
        try {
            LocalDate now = LocalDate.now();
            if (!now.equals(this.currentDate)) {
                this.rotateByDate();
            }
            if (this.maxLogFileSize > 0 && this.currentSize + (long)bytes.length > (long)this.maxLogFileSize) {
                this.rotateBySize();
            }
        }
        catch (IOException e) {
            this.reportError("log rotation has issues : ", e, 1);
        }
    }

    private byte[] getMessageBytes(LogRecord record) {
        byte[] bytes;
        try {
            String msg = this.getFormatter().format(record);
            bytes = msg.getBytes(this.encoding);
        }
        catch (Exception e) {
            this.reportError("Formatting error", e, 5);
            return null;
        }
        return bytes;
    }

    @Override
    public void flush() {
        this.lock.lock();
        try {
            if (this.outputStream != null) {
                this.outputStream.flush();
            }
        }
        catch (IOException e) {
            this.reportError(null, e, 2);
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public void close() throws SecurityException {
        this.lock.lock();
        try {
            if (this.outputStream != null) {
                this.outputStream.close();
            }
        }
        catch (IOException e) {
            this.reportError(null, e, 3);
        }
        finally {
            this.lock.unlock();
        }
    }

    public String getLogFileName() {
        return this.logFileName;
    }

    public int getMaxLogFileSize() {
        return this.maxLogFileSize;
    }

    public boolean isAppend() {
        return this.append;
    }

    public Charset getEncodingUsed() {
        return this.encoding;
    }

    void setCurrentDate(LocalDate currentDate) {
        this.currentDate = currentDate;
    }

    private static /* synthetic */ boolean lambda$cleanupArchives$0(String baseName, File dir, String name) {
        return name.startsWith(baseName) && !name.equals(baseName);
    }
}

